<?php
/**
 * @author     polkurier.pl <api@polkurier.pl>
 * @copyright  polkurier.pl
 * @license    https://opensource.org/licenses/OSL-3.0 OSL/OSL-3.0
 * @category   Module
 * @package    Presta 1.7
 * @subpackage polkurier_import_prestashop
 * @link       https://www.polkurier.pl
 * @since      1.0.0
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

if (!defined('_PS_MODULE_DIR_')) {
    echo '_PS_MODULE_DIR_ is not defined';
}

require_once(_PS_MODULE_DIR_ . "polkurierapimap/autoload.php");

/**
 * Class polkurierapimap
 */
class polkurierapimap extends Module
{
    /**
     * polkurierapimap constructor.
     */
    public function __construct()
    {
        $this->name = 'polkurierapimap';
        $this->tab = 'shipping_logistics';
        $this->version = '1.0.0';
        $this->author = 'polkurier.pl';
        $this->need_instance = 0;
        $this->bootstrap = true;
        $this->ps_versions_compliancy = ['min' => '1.6.1.9', 'max' => '1.7'];
        parent::__construct();
        $this->displayName = $this->l('POLKURIER.PL - IMPORT');
        $this->description = $this->l('Wysyłaj szybciej i wygodniej dzięki integracji z portalem Polkurier.pl. Moduł wspierający import przesyłek dla przewoźnika InPost Paczkomaty 24/7 oraz Paczka w Ruchu');
        $this->confirmUninstall = $this->l('Czy na pewno chcesz odinstalować?');
    }


    /**
     * @return bool
     */
    public function install()
    {
        if (extension_loaded('curl') === false) {
            $this->_errors[] = $this->l('You have to enable the cURL extension on your server to install this module');
            return false;
        }

        if (version_compare(PHP_VERSION, '5.6.0', '<')) {
            $this->_errors[] = 'Zalecana wersja PHP przynajmniej 5.6';
        }

        if (Module::isInstalled('polkurier') === true) {
            $this->_errors[] = 'Przed instalacją proszę usunąć wtyczkę "Polkurier"';
        }

        Sql\PolkurierInstallSql::Install();

        if (!parent::install() || !$this->registerFront()) {
            return false;
        }

        $this->AddCarriers();

        return true;
    }


    /**
     * @return bool
     */
    private function registerFront()
    {
        $validation = true;
        $validation &= $this->registerHook('header');
        if (Tools::version_compare(_PS_VERSION_, '1.7', '>=')) {
            $validation &= $this->registerHook('displayAfterCarrier');
        } else {
            $validation &= $this->registerHook('displayCarrierList');
        }
        return $validation;
    }

    /**
     * @return bool
     */
    public function uninstall()
    {
        foreach (PolkurierApi\ConfigurationKey::$name as $key => $val) {
            Configuration::deleteByName($val);
        }

        Sql\PolkurierUninstallSql::Uninstall();

        return $this->unregisterFront() && parent::uninstall();
    }

    /**
     * @return bool
     */
    private function unregisterFront()
    {
        $validation = true;
        $validation &= $this->unregisterHook('header');
        if (Tools::version_compare(_PS_VERSION_, '1.7', '>=')) {
            $validation &= $this->unregisterHook('displayAfterCarrier');
        } else {
            $validation &= $this->unregisterHook('displayCarrierList');
        }
        return $validation;
    }


    public function AddCarriers()
    {
        $carriers_api_code = [];
        foreach (PolkurierApi\CarriersList::getCarriers() as $item) {
            $id_reference = $this->createCarriers($item);
            if ($id_reference) {
                $carriers_api_code[$item['api_code']][] = $id_reference;
            }
        }
        PolkurierApi\Model\PolkurierConfig::setConfigValue(PolkurierApi\ConfigurationKey::POLKURIER_CARRIERS_API_CODE, json_encode($carriers_api_code));
    }


    /**
     * @param array $param
     * @return false|int
     * @since 1.1.0
     */
    private function createCarriers($param)
    {
        $carrier = new Carrier();
        $carrier->name = $this->l($param['name']);
        $carrier->active = 0;
        $carrier->is_free = 0;
        $carrier->shipping_handling = 1;
        $carrier->shipping_external = 0;
        $carrier->shipping_method = 1;
        $carrier->max_width = 0;
        $carrier->max_height = 0;
        $carrier->max_depth = 0;
        $carrier->max_weight = 0;
        $carrier->grade = 0;
        $carrier->is_module = 1;
        $carrier->need_range = 1;
        $carrier->range_behavior = 1;
        $carrier->external_module_name = $this->name;
        $carrier->url = $param['url'];

        $delay = [];
        foreach (Language::getLanguages(false) as $language) {
            $delay[$language['id_lang']] = $param['name'];
        }
        $carrier->delay = $delay;
        if (!$carrier->save()) {
            return false;
        }
        $groups = [];
        foreach (Group::getGroups((int)Context::getContext()->language->id) as $group) {
            $groups[] = $group['id_group'];
        }
        if (!$carrier->setGroups($groups)) {
            return false;
        }
        copy($this->local_path . 'views/img/' . $param['img'], _PS_SHIP_IMG_DIR_ . (int)$carrier->id . '.jpg');

        $carrier = new Carrier($carrier->id);
        return (int)$carrier->id_reference;
    }


    /**
     * @return array
     */
    private static function getListConfigCarriers()
    {
        return json_decode(PolkurierApi\Model\PolkurierConfig::getConfigValue(PolkurierApi\ConfigurationKey::POLKURIER_CARRIERS_API_CODE), true);
    }

    /**
     * @return array
     */
    private static function getIdReferenceCarriers()
    {
        $reference = [];
        foreach (self::getListConfigCarriers() as $key => $val) {
            if (is_array($val)) {
                foreach ($val as $itemref) {
                    $reference[(int)$itemref] = $key;
                }
            }
        }
        return $reference;
    }

    /**
     * @return bool
     */
    private static function isActiveCarriers()
    {
        $reference = self::getIdReferenceCarriers();
        $activcarrier = false;
        foreach (self::getCarriers() as $item) {
            if ((int)$item['active'] === 1 && (int)$item['deleted'] === 0 && isset($reference[(int)$item['id_reference']]) === true) {
                $activcarrier = true;
            }
        }
        return $activcarrier;
    }

    /**
     * @return array
     */
    private static function getCarriers()
    {
        return Carrier::getCarriers(Context::getContext()->language->id, false, false, false, null, Carrier::ALL_CARRIERS);
    }


    public function hookDisplayAfterCarrier($params)
    {
        $reference = self::getIdReferenceCarriers();
        if (!empty($reference)) {
            try {
                $chosen_collection_point = [];
                $cart = $this->context->cart;
                $carrier_id = $cart->id_carrier;
                $pointData = PolkurierApi\CollectionPoint\CollectionPointModel::getCollectionPointByIdCartIdCarrier((int)$cart->id, (int)$carrier_id);
                if (!empty($pointData)) {
                    $chosen_collection_point[$carrier_id] = $pointData['collection_point'] . '|' . $pointData['adres'];
                }

                $this->context->smarty->assign([
                    'carriers' => $reference,
                    'polkurier_chosen_collection_point' => $chosen_collection_point
                ]);
                return $this->context->smarty->fetch($this->local_path . 'views/templates/front/collection_point.tpl');
            } catch (Exception $ex) {
            }
        }
        return '';
    }


    public function hookDisplayCarrierList($params)
    {
        return $this->hookDisplayAfterCarrier($params);
    }

    /**
     * @param $params
     * @return mixed
     */
    public function hookDisplayCarrierExtraContent($params)///DELLLLLLLLLLLLLLLLLLLLLL
    {

        $reference = self::getIdReferenceCarriers();
        $carrier_id = $params['carrier']['id'];
        if (!empty($reference) && isset($reference[$carrier_id])) {

            try {

                $api_code = $reference[$carrier_id];
                $chosen_collection_point = $searchQuery = '';

                $cart = $this->context->cart;
                $pointData = PolkurierApi\CollectionPoint\CollectionPointModel::getCollectionPointByIdCartIdCarrier((int)$this->context->cart->id, (int)$carrier_id);
                if (!empty($pointData)) {
                    $chosen_collection_point = $pointData['collection_point'] . '|' . $pointData['adres'];
                } else {
                    $deliveryAddress = new Address($cart->id_address_delivery);
                    $searchQuery = $deliveryAddress->postcode . ' ' . $deliveryAddress->city . ' ' . $deliveryAddress->address1;
                }

                $points_map_options = [
                    'providers' => $api_code,
                    'searchQuery' => $searchQuery,
                    'input_chosen' => 'input_chosen_point_' . $api_code,
                    'functions' => ['collect']
                ];

                $this->context->smarty->assign([
                    'polkurier_chosen_collection_point' => $chosen_collection_point,
                    'polkurier_points_map_options' => json_encode($points_map_options),
                    'polkurier_carrier' => $api_code
                ]);
                return $this->context->smarty->fetch($this->local_path . 'views/templates/front/collection_point.tpl');
            } catch (Exception $ex) {
            }
        }
        return '';
    }


    /**
     * Front
     * @return string
     * @return string
     * @throws Exception
     */
    public function hookHeader()
    {
        $controller = isset($this->context->controller) ? $this->context->controller : null;

        $allowed_controllers = array(
            'OrderController',
            'OrderOpcController',
        );

        $is_allowed = !is_null($controller) && in_array(get_class($controller), $allowed_controllers);
        if (!$is_allowed) {
            return '';
        }

        if (self::isActiveCarriers() === false) {
            return '';
        }

        if (Tools::version_compare(_PS_VERSION_, '1.7.0', '>=') === true) {

            $this->context->controller->registerStylesheet(
                'polkurier-remote-points-map-css',
                'https://maps.polkurier.pl/assets/dist/points-map.css',
                [
                    'server' => 'remote',
                    'position' => 'bottom',
                    'priority' => 20
                ]);

            $this->context->controller->registerJavascript(
                'polkurier-remote-points-map-js',
                'https://maps.polkurier.pl/assets/dist/points-map.bundle.js',
                [
                    'priority' => 200,
                    'server' => 'remote',
                    'attribute' => 'async',
                ]);

            $this->context->controller->registerStylesheet(
                'polkurier-parcellocker-css',
                'modules/' . $this->name . '/views/css/polkurier_parcel_locker.css',
                [
                    'media' => 'all',
                    'priority' => 200,
                ]);

            $this->context->controller->registerJavascript(
                'polkurier-collection-point-widget-js',
                'modules/' . $this->name . '/views/js/collection_point_widget.js',
                [
                    'priority' => 200
                ]
            );

            $this->context->controller->registerJavascript(
                'polkurier-collection-point-17-js',
                'modules/' . $this->name . '/views/js/collection_point_17.js',
                [
                    'priority' => 200
                ]
            );

        } else {

            $this->context->controller->addJS('https://maps.polkurier.pl/assets/dist/points-map.bundle.js');
            $this->context->controller->addCSS('https://maps.polkurier.pl/assets/dist/points-map.css');
            $this->context->controller->addJS($this->_path . '/views/js/collection_point_widget.js');
            $this->context->controller->addJS($this->_path . '/views/js/collection_point_16.js');
            $this->context->controller->addCSS($this->_path . '/views/css/polkurier_parcel_locker_16.css');
        }
        return $this->hookHeaderCarrier();
    }


    /**
     * @return string
     * @throws Exception
     */
    public function hookHeaderCarrier()
    {
        $reference = self::getIdReferenceCarriers();

        if (!empty($reference)) {

            $cart = $this->context->cart;
            $AjaxUrl = $this->context->link->getModuleLink($this->name, 'CollectionPoint', ['ajax' => true]);
            $AjaxDeleteUrl = $this->context->link->getModuleLink($this->name, 'DeleteCollectionPoint', ['ajax' => true]);

            $polkurier_map_token = (new PolkurierApi\ApiMetod\GetMapTokenApi())->getToken();
            $polkurier_maps_url = ConfigApiPolkurier::$maps_url;
            $polkurier_collection_point_token = Tools::encrypt($cart->id);

            $deliveryAddress = new Address($cart->id_address_delivery);
            $searchQuery = $deliveryAddress->postcode . ' ' . $deliveryAddress->city . ' ' . $deliveryAddress->address1;
            $points_map_options = [
                'searchQuery' => $searchQuery,
                'functions' => ['collect']
            ];

            $collection_point_option = [
                'ID_CART' => $cart->id,
                'POLKURIER_COLLECTION_POINT_AJAX_URL' => $AjaxUrl,
                'POLKURIER_DELETE_COLLECTION_POINT_AJAX_URL' => $AjaxDeleteUrl,
                'POLKURIER_COLLECTION_POINT_TOKEN' => $polkurier_collection_point_token,
                'POLKURIER_COLLECTION_POINT_API_CODE' => $reference,
                'POLKURIER_MAP_TOKEN' => $polkurier_map_token,
                'POLKURIER_MAPS_URL' => $polkurier_maps_url,
//                'POLKURIER_CHOSEN_COLLECTION_POINT' => $chosen_collection_point,
                'POLKURIER_SEARCH_QUERY' => $searchQuery,
                'POLKURIER_POINTS_MAP_OPTIONS' => $points_map_options
            ];

            if (Tools::version_compare(_PS_VERSION_, '1.7.0', '>=') === true) {

                Media::addJsDef($collection_point_option);

            } else {
                $this->context->smarty->assign($collection_point_option);

                return $this->display(__FILE__, 'views/templates/front/collection_point_front_header.tpl');
            }
        }
        return '';
    }


    /**
     * @param PolkurierApi\Msg $msg
     * @return array
     */
    public function getDisplayErrorHtml(PolkurierApi\Msg $msg)
    {
        $messages = [];
        foreach ($msg->getErrors() as $error) {
            if (!empty($error)) {
                $messages[] = $this->displayError($this->l($error));
            }
        }
        return $messages;
    }

    /**
     * @param PolkurierApi\Msg $msg
     * @return array
     */
    public function getDisplayConfirmationHtml(PolkurierApi\Msg $msg)
    {
        $messages = [];
        foreach ($msg->getConfirmations() as $confirmation) {
            if (!empty($confirmation)) {
                $messages[] = $this->displayConfirmation($this->l($confirmation));
            }
        }
        return $messages;
    }


    /**
     * configuration data
     * @return string|void
     * @throws Exception
     */
    public function getContent()
    {
        if (Tools::isSubmit('savepolkurierauthapiconfiguration')) {
            return $this->AuthDataControl();
        }

        if (Tools::isSubmit('savecollectionpointconfiguration')) {
            return $this->SaveCollectionPointConfiguration();
        }

        return $this->showAdminConfig();
    }

    /**
     * @return mixed
     */
    private function showAdminConfig()
    {
        $this->msg = new \PolkurierApi\Msg();
        $this->context->controller->addJquery();
        $this->context->smarty->assign([
            'updatemodule' => $this->getUpdateModule(),
            'version' => $this->version,
            'module_dir' => $this->_path
        ]);

        /* nagłówek */
        $head = $this->context->smarty->fetch($this->local_path . PolkurierAdminConfigTopView::getTemplate());

        /* dane autoryzacyjne */
        $output = $this->renderAuthForm();

        /**/
        $output .= $this->getFormCarriers();

        $variables['module_dir'] = $this->_path;
        $variables['base_url'] = $this->context->link->getAdminLink('AdminModules') . '&configure=' . $this->name;
        $variables['polkurier_token'] = Tools::getAdminTokenLite('AdminPolkurierModule');

        $this->context->smarty->assign($variables);

        $infohtml = '';
        $confirmation = $this->getDisplayConfirmationHtml($this->msg);
        if (is_array($confirmation)) {
            foreach ($confirmation as $item) {
                $infohtml .= $item;
            }
        }

        $error = $this->getDisplayErrorHtml($this->msg);
        if (is_array($error)) {
            foreach ($error as $item) {
                $infohtml .= $item;
            }
        }

        return $head . $infohtml . $output;
    }


    public function getFormCarriers()
    {
        $carriers = Carrier::getCarriers(
            $this->context->language->id,
            true,
            false,
            false,
            null,
            Carrier::ALL_CARRIERS
        );

        $helper = new HelperForm();
        $helper->tpl_vars['fields_value']['info'] = '<div>' . $this->l('Wybierz mapę punktów, którą przypiszemy dla poszczególnych przewoźników.') . '</div>';

        $hookcarriers = self::getListConfigCarriers();

        foreach (PolkurierApi\CarriersList::getCarriers() as $item) {
            $inputs[] = [
                'label' => $item['label'],
                'name' => $item['api_code'] . '[]',
                'options' => [
                    'query' => $carriers,
                    'id' => 'id_reference',
                    'name' => 'name'
                ],
                'type' => 'select',
                'class' => 'chosen',
                'multiple' => true
            ];
            $fields_value = [];
            if (isset($hookcarriers[$item['api_code']]) && !empty($hookcarriers[$item['api_code']])) {
                $fields_value = $hookcarriers[$item['api_code']];
            }
            $helper->tpl_vars['fields_value'][$item['api_code'] . '[]'] = $fields_value;
        }

        $fields_form = [
            'form' => [
                'legend' => [
                    'tinymce' => false,
                    'title' => $this->l('Konfiguracja map'),
                    'icon' => 'icon-cogs'
                ],
                'input' => $inputs,
                'submit' => [
                    'title' => $this->l('Zapisz'),
                ],
            ],
        ];


        $helper->show_toolbar = false;
        $helper->table = 'polkurier_collection_point';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->submit_action = 'savecollectionpointconfiguration';
        $helper->identifier = $this->identifier;

        return $helper->generateForm(array($fields_form));
    }

    private function redirectToAdminConfig()
    {
        Tools::redirectAdmin(Context::getContext()->link->getAdminLink('AdminModules') . '&configure=' . $this->name);
    }

    /**
     * @return bool|string
     */
    private static function getAdminModulesToken()
    {
        return Tools::getAdminTokenLite('AdminModules');
    }


    /**
     * authorization data
     */
    private function AuthDataControl()
    {
        PolkurierApi\Model\PolkurierConfig::setConfigValue(
            PolkurierApi\ConfigurationKey::POLKURIER_AUTH_API_ACCOUNT,
            Tools::getValue(PolkurierApi\ConfigurationKey::POLKURIER_AUTH_API_ACCOUNT, '')
        );
        PolkurierApi\Model\PolkurierConfig::setConfigValue(
            PolkurierApi\ConfigurationKey::POLKURIER_AUTH_API_TOKEN,
            Tools::getValue(PolkurierApi\ConfigurationKey::POLKURIER_AUTH_API_TOKEN, '')
        );
        $this->redirectToAdminConfig();
    }


    private function SaveCollectionPointConfiguration()
    {
        $carriers = [];
        foreach (PolkurierApi\CarriersList::getCarriers() as $item) {

            $carriers[$item['api_code']] = Tools::getValue($item['api_code'], []);
        }
        PolkurierApi\Model\PolkurierConfig::setConfigValue(PolkurierApi\ConfigurationKey::POLKURIER_CARRIERS_API_CODE, json_encode($carriers));
        $this->redirectToAdminConfig();
    }


    /**
     * @return string
     * @throws PrestaShopException
     */
    private function renderAuthForm()
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = 'polkurierauthapiconfiguration';
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = self::getAdminModulesToken();
        $helper->submit_action = 'savepolkurierauthapiconfiguration';

        $helper->tpl_vars = array(
            'fields_value' => [
                'POLKURIER_AUTH_API_ACCOUNT' => PolkurierApi\Model\PolkurierConfig::getConfigValue(PolkurierApi\ConfigurationKey::POLKURIER_AUTH_API_ACCOUNT),
                'POLKURIER_AUTH_API_TOKEN' => PolkurierApi\Model\PolkurierConfig::getConfigValue(PolkurierApi\ConfigurationKey::POLKURIER_AUTH_API_TOKEN),
            ],
        );

        try {

            PolkurierApi\Model\PolkurierConfig::CheckIsDefined();
            $apiclient = new PolkurierApi\ApiMetod\ApiClient();
            $apiclient->checkAPI();

        } catch (Exception $ex) {
            $this->msg->setErrors($ex->getMessage());
        }
        return $helper->generateForm(array(PolkurierConfigAuthView::getFormConfig()));
    }


    /**
     * @return string
     */
    public static function uniqhash()
    {
        return md5(uniqid(microtime(true), true));
    }

    /**
     * @return array
     */
    private function getUpdateModule()
    {
        $output = [
            'newversion' => false,
            'newupdate' => false
        ];
        try {
            $file = Tools::file_get_contents(ConfigApiPolkurier::$url_updatemodule, true, null, 10);
            if ($file) {
                $json = json_decode($file, true);
                if (Tools::version_compare($this->version, $json['version'], '<') === true) {
                    $output['newversion'] = true;
                    $output['version'] = $json['version'];
                    $output['description'] = $json['description'];
                    $output['version_date'] = $json['version_date'];
                }
            }
        } catch (Exception $exp) {
        }
        return $output;
    }
}
