<?php

namespace PolkurierWebServiceApi\Methods;

use PolkurierWebServiceApi\Entities\COD;
use PolkurierWebServiceApi\Entities\CourierServiceInterface;
use PolkurierWebServiceApi\Entities\Pack;
use PolkurierWebServiceApi\Entities\Pickup;
use PolkurierWebServiceApi\Entities\Recipient;
use PolkurierWebServiceApi\Entities\Sender;
use PolkurierWebServiceApi\Entities\SenderCoverAddress;

class CreateOrder extends AbstractMethod
{
    private string $shipmenttype = '';

    private string $courier = '';

    private array $courierservice = [];

    private string $description = '';

    private Sender $sender;

    private Recipient $recipient;

    private array $packs = [];

    private Pickup $pickup;

    private COD $COD;

    private float $insurance = 0.0;

    private int $platform = 0;

    private ?SenderCoverAddress $senderCoverAddress = null;

    public function __construct()
    {
        $this->sender = new Sender();
        $this->recipient = new Recipient();
        $this->COD = new COD();
        $this->pickup = new Pickup();
    }

    public function getName(): string
    {
        return 'create_order';
    }

    public function getRequestData(): array
    {
        return [
            'shipmenttype' => $this->shipmenttype,
            'courier' => $this->courier,
            'courierservice' => $this->getCourierServiceMap(),
            'description' => $this->description,
            'sender' => $this->sender->toArray(),
            'recipient' => $this->recipient->toArray(),
            'packs' => $this->getPack(),
            'pickup' => $this->pickup->toArray(),
            'COD' => $this->COD->toArray(),
            'insurance' => $this->insurance,
            'platform' => $this->getPlatform(),
            'cover_address' => $this->senderCoverAddress?->toArray()
        ];
    }

    private function getPack(): array
    {
        return array_map(static function (Pack $pack) {
            return $pack->toArray();
        }, $this->packs);
    }

    private function getCourierServiceMap(): array
    {
        $servicemap = [];
        foreach ($this->courierservice as $item) {
            $itemarray = $item->toArray();
            $servicemap[key($itemarray)] = current($itemarray);
        }
        return $servicemap;
    }

    public function getPlatform(): int
    {
        return $this->platform;
    }

    public function setPlatform($platform): CreateOrder
    {
        $this->platform = $platform;
        return $this;
    }

    public function setCourier(string $courier = ''): CreateOrder
    {
        $this->courier = $courier;
        return $this;
    }

    public function setShipmentType(string $shipmenttype = ''): CreateOrder
    {
        $this->shipmenttype = $shipmenttype;
        return $this;
    }

    public function setDescription(string $description = ''): CreateOrder
    {
        $this->description = $description;
        return $this;
    }

    public function setSender(Sender $sender): CreateOrder
    {
        $this->sender = $sender;
        return $this;
    }

    public function setRecipient(Recipient $recipient): CreateOrder
    {
        $this->recipient = $recipient;
        return $this;
    }

    public function setInsurance($insurance): CreateOrder
    {
        $this->insurance = (float)$insurance;
        return $this;
    }

    public function addPack(Pack $pack): CreateOrder
    {
        $this->packs[] = $pack;
        return $this;
    }

    public function addCourierService(CourierServiceInterface $courierservice): CreateOrder
    {
        $this->courierservice[] = $courierservice;
        return $this;
    }

    public function setPickup(Pickup $pickup): CreateOrder
    {
        $this->pickup = $pickup;
        return $this;
    }

    public function setCOD(COD $cod): CreateOrder
    {
        $this->COD = $cod;
        return $this;
    }

    public function setSenderCoverAddress(?SenderCoverAddress $senderCoverAddress): void
    {
        $this->senderCoverAddress = $senderCoverAddress;
    }
}