<?php
/**
 * @author     polkurier.pl <api@polkurier.pl>
 * @copyright  polkurier.pl
 * @license    https://opensource.org/licenses/OSL-3.0 OSL/OSL-3.0
 * @category   Module
 * @package    Presta 8
 * @subpackage polkurier
 * @link       https://www.polkurier.pl
 */

namespace Polkurier\Model\Shipment\NewShipment;

use Polkurier\Model\Shipment\NewShipment\Module\DeliveryAddress\DeliveryAddressModule;
use Polkurier\Model\Shipment\Receiver;
use Address;
use Customer;
use Order;
use Exception;
use RuntimeException;

final class DeliveryAddressModel
{
    private int $id_order;

    public function __construct(int $id_order)
    {
        $this->id_order = $id_order;
    }

    /**
     * Adres dostawy
     */
    public function getDeliveryAddress(): Receiver
    {
        if (!$this->id_order) {

            throw new RuntimeException('Value is empty: ID zamówienia');
        }
        
        $order = new Order($this->id_order);
        
        if (!is_numeric($order->id_address_delivery)) {

            throw new RuntimeException('Value is empty: ID adres dostawy');
        }
        $address_id = (int)$order->id_address_delivery;
        
        return $this->updateDeliveryAddress(new Address($address_id), new Customer($order->id_customer));
    }


    private function updateDeliveryAddress(Address $address, Customer $customer): Receiver
    {
        $receiver = new Receiver();
        $receiver->setCompany($address->company);
        $receiver->setAddress1($address->address1);
        $receiver->setAddress2($address->address2);
        $receiver->setPostcode($address->postcode);
        $receiver->setCity($address->city);

        if ($address->id_country) {
            $receiver->setCountryId($address->id_country);
        }
        
        $receiver->setFirstname($address->firstname);
        $receiver->setLastname($address->lastname);
        
        if (empty($address->phone)) {

            $receiver->setPhone($address->phone_mobile);

        } else {

            $receiver->setPhone($address->phone);
        }

        $receiver->setEmail($customer->email);
        
        $this->updateReceiverStreetNofromAddress($receiver);
        
        (new DeliveryAddressModule($this->id_order, $receiver))->setAddress();
        
        return $receiver;
    }
    
    
    /**
     * Rozbija adres odbiorcy na ulicę i numer
     */
    private function updateReceiverStreetNofromAddress(Receiver $receiver): void
    {
        $street = trim($receiver->getAddress1() . ' ' . $receiver->getAddress2());
        if (!empty($street)) {
            $splitstreet = self::SplitStreet($street);

            if ($splitstreet['validate']) {

                $receiver->setStreet(trim($splitstreet['street']));
                $receiver->setHouseNo(trim($splitstreet['number']));
                $receiver->setFlatNo($splitstreet['numberAddition']);

            } else {

                $receiver->setStreet(trim(preg_replace('/(\s{2,})/', ' ', $street)));
            }
        }
    }

    public static function SplitStreet($streetstr): array
    {
        $streetstr = trim(preg_replace('/(\s{2,})/', ' ', $streetstr));
        $streetstr = trim(str_replace("ul.", "", $streetstr));
        $amatch = [];
        $pattern = '/^([\w[:punct:] ]+) ([0-9]{1,5})([\w[:punct:]\-\/]*)$/u';
        preg_match($pattern, $streetstr, $amatch);
        $street = (isset($amatch[1])) ? trim($amatch[1]) : '';
        $number = (isset($amatch[2])) ? trim($amatch[2]) : '';
        $numberAddition = (isset($amatch[3])) ? trim($amatch[3]) : '';
        
        if (empty($street) || empty($number)) {
            return [
                'validate' => false
            ];
        }
        return [
            'street' => $street,
            'number' => $number,
            'numberAddition' => str_replace("/", "", $numberAddition),
            'validate' => true
        ];
    }
}