<?php

namespace PolkurierWebServiceApi\Methods;

use PolkurierWebServiceApi\Entities\CourierServiceInterface;
use PolkurierWebServiceApi\Entities\Pack;
use PolkurierWebServiceApi\Response;
use PolkurierWebServiceApi\Type\ReturnCodType;
use PolkurierWebServiceApi\Type\ReturnTimeCodType;
use PolkurierWebServiceApi\Util\Arr;
use PolkurierWebServiceApi\Entities\OrderValuation as Valuation;

/**
 * Class OrderValuation
 * @package PolkurierWebServiceApi\Methods
 *
 */
class OrderValuation extends AbstractMethod
{
    /**
     * @var string
     */
    private $returnValuations = '';
    /**
     * @var string
     */
    private $shipmentType = '';
    /**
     * @var array
     */
    private $packs = [];
    /**
     * @var float
     */
    private $COD = 0.0;
    /**
     * @var float
     */
    private $insurance = 0.0;
    /**
     * Czas zwrotu pobrania
     * @var string
     */
    private $codtype = ReturnTimeCodType::DEFAULT_TYPE;
    /**
     * @var string
     */
    private $return_cod = ReturnCodType::DEFAULT_TYPE;
    /**
     * @var string
     */
    private $postcode_recipient;
    /**
     * @var string
     */
    private $postcode_sender;
    /**
     * @var string
     */
    private $recipient_country;
    /**
     * @var CourierServiceInterface[]
     */
    private $courierservice = [];
    
    /**
     * @return string
     */
    public function getName()
    {
        return 'order_valuation';
    }
    
    /**
     * @return string
     */
    public function getPostcodeRecipient()
    {
        return $this->postcode_recipient;
    }
    
    /**
     * @param string $postcode_recipient
     * @return $this
     */
    public function setPostcodeRecipient($postcode_recipient)
    {
        $this->postcode_recipient = $postcode_recipient;
        return $this;
    }
    
    /**
     * @return string
     */
    public function getPostcodeSender()
    {
        return $this->postcode_sender;
    }
    
    /**
     * @param string $postcode_sender
     * @return $this
     */
    public function setPostcodeSender($postcode_sender)
    {
        $this->postcode_sender = $postcode_sender;
        return $this;
    }
    
    /**
     * @return string
     */
    public function getRecipientCountry()
    {
        return $this->recipient_country;
    }
    
    /**
     * @param string $recipient_country
     * @return $this
     */
    public function setRecipientCountry($recipient_country)
    {
        $this->recipient_country = $recipient_country;
        return $this;
    }

    /**
     * @return string
     */
    public function getReturnCod()
    {
        return $this->return_cod;
    }
    
    /**
     * @param string $return_cod
     * @return $this
     */
    public function setReturnCod($return_cod = '')
    {
        if (!empty($return_cod)) {
            $this->return_cod = $return_cod;
        }
        return $this;
    }
    
    /**
     * @return string
     */
    public function getCodtype()
    {
        return $this->codtype;
    }
    
    /**
     * @param string $codtype
     * @return $this
     */
    public function setCodtype($codtype = '')
    {
        if (!empty($codtype)) {
            $this->codtype = $codtype;
        }
        return $this;
    }


    /**
     * @param Pack $pack
     * @return $this
     */
    public function addPack(Pack $pack)
    {
        $this->packs[] = $pack;
        return $this;
    }
    
    /**
     * @param string $returnValuations
     * @return OrderValuation
     */
    public function setReturnValuations($returnValuations)
    {
        $this->returnValuations = (string)$returnValuations;
        return $this;
    }
    

    /**
     * @param string $shipmentType
     * @return OrderValuation
     */
    public function setShipmentType($shipmentType)
    {
        $this->shipmentType = (string)$shipmentType;
        return $this;
    }

    
    /**
     * @param float $COD
     * @return OrderValuation
     */
    public function setCOD($COD)
    {
        $this->COD = $COD;
        return $this;
    }
    

    /**
     * @param float $insurance
     * @return OrderValuation
     */
    public function setInsurance($insurance)
    {
        $this->insurance = $insurance;
        return $this;
    }

    /**
     * @param CourierServiceInterface $courierservice
     * @return $this
     */
    public function addCourierService(CourierServiceInterface $courierservice)
    {
        $this->courierservice[] = $courierservice;
        return $this;
    }

    /**
     * @return array
     */
    private function getCourierServiceMap()
    {
        $servicemap = [];
        foreach ($this->courierservice as $item) {
            $itemarray = $item->toArray();
            $servicemap[key($itemarray)] = current($itemarray);
        }
        return $servicemap;
    }

    /**
     * @return array
     */
    public function getRequestData()
    {
        return [
            'returnvaluations' => $this->returnValuations,
            'shipmenttype' => $this->shipmentType,
            'packs' => array_map(function (Pack $pack) {
                return $pack->toArray();
            }, $this->packs),
            'COD' => $this->COD,
            'codtype' => $this->getCodtype(),
            'return_cod' => $this->getReturnCod(),
            'insurance' => $this->insurance,
            'postcode_recipient' => $this->getPostcodeRecipient(),
            'postcode_sender' => $this->getPostcodeSender(),
            'recipient_country' => $this->getRecipientCountry(),
            'courierservice' => $this->getCourierServiceMap(),
        ];
    }

    /**
     * @param Response $response
     * @return $this|OrderValuation
     */
    public function setResponseData(Response $response)
    {
        $this->responseData = [];
        $rawData = $response->get('response');
        
        if ($this->returnValuations) {
            $rawData = [$rawData];
        }
        
        foreach ($rawData as $row) {
            $item = new Valuation();
            $item->setGrossPrice(Arr::get($row, 'grossprice'));
            $item->setNetPrice(Arr::get($row, 'netprice'));
            $item->setServiceCode(Arr::get($row, 'servicecode'));
            $item->setServiceName(Arr::get($row, 'serviceName'));
            $item->setShipment(Arr::get($row, 'shipment'));
            $this->responseData[] = $item;
        }
        return $this;
    }
}