<?php
/**
 * @author     polkurier.pl <api@polkurier.pl>
 * @copyright  polkurier.pl
 * @license    https://opensource.org/licenses/OSL-3.0 OSL/OSL-3.0
 * @category   Module
 * @package    Presta 8
 * @subpackage polkurier
 * @link       https://www.polkurier.pl
 */

namespace Polkurier\Model\Shipment\FormShipment;

use Polkurier\Bank\BankListModel;
use Polkurier\Model\PolkurierConfig;
use Polkurier\Model\Shipment\Receiver;
use Polkurier\Model\Shipment\Sender;
use Polkurier\Model\Shipment\SenderCoverAddress;
use Polkurier\Model\Shipment\ShipmenPack;
use Polkurier\Model\Shipment\ShipmentAdditionalServices;
use Polkurier\Model\Shipment\ShipmentModel;
use Polkurier\CoverAddress\SqlSenderCoverAddress;
use Db;
use Polkurier\Model\SqlAbstract;
use Polkurier\Model\CountryModel;

class FormShipmentModel
{
    private ShipmentModel $shipmentmodel;
    private array $formdata;

    public function __construct(ShipmentModel $shipmentmodel, array $formdata = [])
    {
        $this->shipmentmodel = $shipmentmodel;
        $this->formdata = $formdata;
    }

    /**
     * Dane z formularza
     */
    public function SetModel(): void
    {
        $this->shipmentmodel->setPrestaIdOrder($this->formdata['presta_id_order']);

        /* the recipient's address */
        $this->shipmentmodel->setReceiver($this->DeliveryAddressFromForm());

        /* the sender's address */
        $this->shipmentmodel->setSender($this->DeliverySenderFromForm());

        $this->shipmentmodel->setSenderCoverAddress($this->getCoverAddressFromForm());

        /* description of the content */
        if (self::getValArray($this->formdata, 'polkurier_order_content')) {
            $this->shipmentmodel->setOrderContent(self::getValArray($this->formdata, 'polkurier_order_content'));
        }

        /* insurance */
        if ((float)self::getValArray($this->formdata, 'polkurier_insurance_value') > 0) {
            $this->shipmentmodel->setInsuranceValue((float)self::getValArray($this->formdata, 'polkurier_insurance_value'));
        }

        /* type of delivery */
        if (self::getValArray($this->formdata, 'polkurier_shipment_type')) {
            $this->shipmentmodel->setShipmentType(self::getValArray($this->formdata, 'polkurier_shipment_type'));
        }

        /* carrier */
        if (self::getValArray($this->formdata, 'poklurier_service_code')) {
            $this->shipmentmodel->setServiceCode(self::getValArray($this->formdata, 'poklurier_service_code'));
        }

        $this->setCOD();

        $this->setPack();

        $this->setPickup();

        $this->setOrderAdditionalServices();
    }


    /**
     * Pickup
     */
    private function setPickup(): void
    {
        if (self::getValArray($this->formdata, 'polkurier_pickup_type')) {

            if (isset(PolkurierConfig::$pickup_courier_type[self::getValArray($this->formdata, 'polkurier_pickup_type')])) {

                $pickup_type = PolkurierConfig::$pickup_courier_type[self::getValArray($this->formdata, 'polkurier_pickup_type')];

                if (!$pickup_type) {

                    $this->shipmentmodel->getShipmentPickup()->setNocourier(true);
                }
            }
        }

        if ($pickup_type) {

            if (self::getValArray($this->formdata, 'polkurier_date_pickup')) {

                $this->shipmentmodel->getShipmentPickup()->setDate(
                    (string)self::getValArray($this->formdata, 'polkurier_date_pickup')
                );
            }

            if (self::getValArray($this->formdata, 'polkurier_time_pickup')) {

                $time_pickup = explode('|', self::getValArray($this->formdata, 'polkurier_time_pickup'));
                $this->shipmentmodel->getShipmentPickup()->setTimefrom((string)$time_pickup[0]);
                $this->shipmentmodel->getShipmentPickup()->setTimeto((string)$time_pickup[1]);
            }
        }
    }

    /**
     * COD
     */
    private function setCOD(): void
    {
        /* cash on delivery refund amount */
        if ((float)self::getValArray($this->formdata, 'polkurier_cod_order') > 0) {

            $this->shipmentmodel->getCodShipment()->setValue((float)self::getValArray($this->formdata, 'polkurier_cod_order'));
        }

        /* method of returning the COD */
        if (self::getValArray($this->formdata, 'polkurier_return_cod_type')) {

            $this->shipmentmodel->getCodShipment()->setType(self::getValArray($this->formdata, 'polkurier_return_cod_type'));
        }

        /* COD return time */
        if (self::getValArray($this->formdata, 'polkurier_return_cod_time')) {

            $this->shipmentmodel->getCodShipment()->setTime(self::getValArray($this->formdata, 'polkurier_return_cod_time'));
        }

        /* Bank account number */
        if (self::getValArray($this->formdata, 'polkurier_sender_bank_id')) {

            $this->shipmentmodel->getCodShipment()->setIdCod(self::getValArray($this->formdata, 'polkurier_sender_bank_id'));
            $this->shipmentmodel->getCodShipment()->setNumber(BankListModel::getNumberAccountBank(self::getValArray($this->formdata, 'polkurier_sender_bank_id')));
        }
    }

    /**
     * the recipient's details
     */
    private function DeliveryAddressFromForm(): Receiver
    {
        $receiver_form = $this->formdata['receiver'];
        $receiver = new Receiver();

        if (is_array($receiver_form)) {

            $receiver->setCompany(self::getValArray($receiver_form, 'company'));
            $receiver->setStreet(self::getValArray($receiver_form, 'street'));
            $receiver->setHouseNo(self::getValArray($receiver_form, 'house_no'));
            $receiver->setFlatNo(self::getValArray($receiver_form, 'flat_no'));
            $receiver->setPostcode(self::getValArray($receiver_form, 'postcode'));
            $receiver->setCity(self::getValArray($receiver_form, 'city'));

            if (!empty(self::getValArray($receiver_form, 'country'))) {
                $countryiso = self::getValArray($receiver_form, 'country');
                $receiver->setCountryId(CountryModel::getIdByIso($countryiso));
                $receiver->setCountryIso($countryiso);
            }

            $receiver->setFirstname(self::getValArray($receiver_form, 'firstname'));
            $receiver->setLastname(self::getValArray($receiver_form, 'lastname'));
            $receiver->setPhone(self::getValArray($receiver_form, 'phone'));
            $receiver->setEmail(self::getValArray($receiver_form, 'email'));
            $receiver->setStation(self::getValArray($receiver_form, 'station'));
        }

        return $receiver;
    }


    /**
     * the sender's address
     */
    private function DeliverySenderFromForm(): Sender
    {
        $sender_form = $this->formdata['sender'];
        $sender = new Sender();

        if (is_array($sender_form)) {

            $sender->setCompany(self::getValArray($sender_form, 'company'));
            $sender->setFirstname(self::getValArray($sender_form, 'firstname'));
            $sender->setContactName(self::getValArray($sender_form, 'contact_name'));
            $sender->setLastname(self::getValArray($sender_form, 'lastname'));
            $sender->setStreet(self::getValArray($sender_form, 'street'));
            $sender->setHouseNo(self::getValArray($sender_form, 'house_no'));
            $sender->setFlatNo(self::getValArray($sender_form, 'flat_no'));
            $sender->setPostcode(self::getValArray($sender_form, 'postcode'));
            $sender->setCity(self::getValArray($sender_form, 'city'));
            $sender->setPhone(self::getValArray($sender_form, 'phone'));
            $sender->setEmail(self::getValArray($sender_form, 'email'));
            $sender->setStation(self::getValArray($sender_form, 'station'));
        }
        return $sender;
    }


    private function getCoverAddressFromForm()
    {
        $senderForm = $this->formdata['sender'];
        $coverAddressId = (int)self::getValArray($senderForm, 'cover_address_template_id');

        if ($coverAddressId !== 0) {

            $coverAddressData = SqlSenderCoverAddress::getSenderCoverAddressById($coverAddressId);

            if (!empty($coverAddressData)) {

                $coverAddressObject = new SenderCoverAddress();
                $coverAddressObject->setCompany(self::getValArray($coverAddressData, 'company'))
                    ->setFirstname(self::getValArray($coverAddressData, 'first_name'))
                    ->setLastname(self::getValArray($coverAddressData, 'last_name'))
                    ->setStreet(self::getValArray($coverAddressData, 'street'))
                    ->setHouseNo(self::getValArray($coverAddressData, 'house_no'))
                    ->setFlatNo(self::getValArray($coverAddressData, 'flat_no'))
                    ->setPostcode(self::getValArray($coverAddressData, 'postcode'))
                    ->setCity(self::getValArray($coverAddressData, 'city'))
                    ->setEmail(self::getValArray($coverAddressData, 'email'))
                    ->setPhone(self::getValArray($coverAddressData, 'phone'));

                return $coverAddressObject;
            }
            return null;
        }
        return null;
    }


    /**
     * collection of packages in the shipment
     */
    public function setPack(): void
    {
        $parcel_collection = self::getValArray($this->formdata, 'polkuier_parcel_collection');

        if (is_array($parcel_collection)) {

            foreach ($parcel_collection as $item) {

                $pack = new ShipmenPack();

                if (self::getValArray($item, 'dim_1')) {
                    $pack->setLength((float)self::getValArray($item, 'dim_1'));
                }

                if (self::getValArray($item, 'dim_2')) {
                    $pack->setWidth((float)self::getValArray($item, 'dim_2'));
                }

                if (self::getValArray($item, 'dim_3')) {
                    $pack->setHeight((float)self::getValArray($item, 'dim_3'));
                }

                if (self::getValArray($item, 'weight')) {
                    $pack->setWeight((float)self::getValArray($item, 'weight'));
                }

                if (self::getValArray($item, 'quantity')) {
                    $pack->setAmount((int)self::getValArray($item, 'quantity'));
                }

                if ((int)self::getValArray($item, 'no_std') === 1) {
                    $pack->setType('NST');
                }

                $this->shipmentmodel->setShipmentPack($pack);
            }
        }
    }


    private function setOrderAdditionalServices(): void
    {
        $service = self::getValArray($this->formdata, 'polkurier_order_service');

        if (isset($service['sms_notification_recipient']) && (int)$service['sms_notification_recipient'] === 1) {

            $this->shipmentmodel->setShipmentadditionalservices(ShipmentAdditionalServices::SMS_NOTIFICATION_RECIPIENT);
        }

        if (isset($service['printed_label']) && (int)$service['printed_label'] === 1) {

            $this->shipmentmodel->setShipmentadditionalservices(ShipmentAdditionalServices::COURIER_PRINTED_LABEL);
        }

        if (isset($service['week_collection']) && (int)$service['week_collection'] === 1) {

            $this->shipmentmodel->setShipmentadditionalservices(ShipmentAdditionalServices::COURIER_WEEK_COLLECTION);
        }
    }


    private static function getValArray(array $data = [], string $key = '')
    {
        if (!empty($key) && isset($data[$key])) {
            return $data[$key];
        }
        return null;
    }
}