<?php
/**
 * @author     polkurier.pl <api@polkurier.pl>
 * @copyright  polkurier.pl
 * @license    https://opensource.org/licenses/OSL-3.0 OSL/OSL-3.0
 * @category   Module
 * @package    Presta 8
 * @subpackage polkurier
 * @link       https://www.polkurier.pl
 */

namespace Polkurier\Model\Shipment\NewShipment;

use Polkurier\ConfigurationKey;
use Polkurier\Model\PolkurierConfig;
use Polkurier\Model\Shipment\CashonDelivery;
use Polkurier\Model\Shipment\ShipmentModel;
use Order;

class NewShipmentModel
{
    private ShipmentModel $shipmentmodel;

    /**
     * Id zamówienia w sklepie
     */
    private $id_order;

    public function __construct(ShipmentModel $shipmentmodel)
    {
        $this->shipmentmodel = $shipmentmodel;
    }

    public function setIdOrder(int $id_order): NewShipmentModel
    {
        $this->id_order = $id_order;
        return $this;
    }

    /**
     * Dane dla nowego zamówienia
     */
    public function NewOrder(): void
    {
        $this->shipmentmodel->setPrestaIdOrder($this->id_order);

        /* Adres odbiorcy */
        $this->shipmentmodel->setReceiver((new DeliveryAddressModel($this->id_order))->getDeliveryAddress());

        /* Adres nadawcy */
        $this->shipmentmodel->setSender((new SenderAddressModel())->getSenderAddress());

        /* Domyślny rodzaj przesyłki */
        $this->shipmentmodel->setShipmentType(PolkurierConfig::$default_shipment_type);

        /* Domyślny id banku */
        if (PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_BANK_ID)) {
            $this->shipmentmodel->getCodShipment()->setIdCod(PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_BANK_ID));
        }

        /* Domyślny rodzaj zwrotu */
        if (PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_RETURN_COD_TYPE)) {
            $this->shipmentmodel->getCodShipment()->setType(PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_RETURN_COD_TYPE));
        }

        $order = new Order($this->id_order);
        if ($this->isCOD($order) === true) {
            $codvalue = $this->getCodValue($order);
            $this->shipmentmodel->getCodShipment()->setValue($codvalue);
            $this->shipmentmodel->setInsuranceValue($codvalue);
            if (PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_RETURN_TIME_COD)) {
                $this->shipmentmodel->getCodShipment()->setTime(PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_RETURN_TIME_COD));
            }
        }
        $this->shipmentmodel->setOrderContent(sprintf('Zamówienie nr: %s', $order->reference));

        /* default carrier */
        if (PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_COURIER)) {
            $this->shipmentmodel->setServiceCode(PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_COURIER));
        }
    }

    private function isCOD(Order $order): bool
    {
        return !(null === $order->module || !in_array($order->module, CashonDelivery::$cashondeliverymodule, true));
    }


    private function getCodValue(Order $order): float
    {
        $total_paid_tax_incl = (float)number_format($order->total_paid_tax_incl, 2, '.', '');
        if (in_array($order->module, CashonDelivery::$cashondeliverymodule, true)) {
            return $total_paid_tax_incl;
        }
        return (float)max(0, $total_paid_tax_incl - $this->getTotalPaid($order));
    }

    private function getTotalPaid(Order $order): float
    {
        return (float)number_format($order->getTotalPaid(new \Currency(\Currency::getIdByIsoCode('PLN'))), 2, '.', '');
    }
}