<?php
/**
 * @author     polkurier.pl <api@polkurier.pl>
 * @copyright  polkurier.pl
 * @license    https://opensource.org/licenses/OSL-3.0 OSL/OSL-3.0
 * @category   Module
 * @package    Presta 1.7
 * @subpackage polkurier
 * @link       https://www.polkurier.pl
 */

namespace Polkurier\Model\Shipment\NewShipment;

use Polkurier\Model\Shipment\NewShipment\Module\DeliveryAddress\DeliveryAddressModule;
use Polkurier\Model\Shipment\Receiver;
use Address;
use Customer;
use Order;
use Exception;

final class DeliveryAddressModel
{
    /**
     * @var int
     */
    private $id_order;
    
    /**
     * @param int $id_order
     */
    public function __construct($id_order)
    {
        $this->id_order = (int)$id_order;
    }

    /**
     * Adres dostawy
     * @return Receiver
     * @throws Exception
     */
    public function getDeliveryAddress()
    {
        if (!is_numeric($this->id_order)) {
            throw new Exception('Value is empty: ID zamówienia');
        }
        
        $order = new Order($this->id_order);
        
        if (!is_numeric($order->id_address_delivery)) {
            throw new Exception('Value is empty: ID adres dostawy');
        }
        $address_id = (int)$order->id_address_delivery;
        
        return $this->updateDeliveryAddress(new Address($address_id), new Customer($order->id_customer));
    }
    
    /**
     * @param \Address $address
     * @param \Customer $customer
     * @return Receiver
     */
    private function updateDeliveryAddress(Address $address, Customer $customer)
    {
        $receiver = new Receiver();
        $receiver->setCompany($address->company);
        $receiver->setAddress1($address->address1);
        $receiver->setAddress2($address->address2);
        $receiver->setPostcode($address->postcode);
        $receiver->setCity($address->city);
        
        if (is_numeric($address->id_country)) {
            $receiver->setCountry($address->id_country);
        }
        
        $receiver->setFirstname($address->firstname);
        $receiver->setLastname($address->lastname);
        
        if (empty($address->phone)) {
            $receiver->setPhone($address->phone_mobile);
        } else {
            $receiver->setPhone($address->phone);
        }
        $receiver->setEmail($customer->email);
        
        $this->updateReceiverStreetNofromAddress($receiver);
        
        (new DeliveryAddressModule($this->id_order, $receiver))->setAddress();
        
        return $receiver;
    }
    
    
    /**
     * Rozbija adres odbiorcy na ulicę i numer
     * @param Receiver $receiver
     */
    private function updateReceiverStreetNofromAddress(Receiver $receiver)
    {
        $street = trim($receiver->getAddress1() . ' ' . $receiver->getAddress2());
        if (!empty($street)) {
            $splitstreet = self::SplitStreet($street);
            if ($splitstreet['validate']) {
                $receiver->setStreet(trim($splitstreet['street']));
                $receiver->setHouseNo(trim($splitstreet['number']));
                $receiver->setFlatNo($splitstreet['numberAddition']);
            } else {
                $receiver->setStreet(trim(preg_replace('/(\s{2,})/', ' ', $street)));
            }
        }
    }
    
    /**
     * @param string $streetstr
     * @return array
     */
    public static function SplitStreet($streetstr)
    {
        $streetstr = trim(preg_replace('/(\s{2,})/', ' ', $streetstr));
        $streetstr = trim(str_replace("ul.", "", $streetstr));
        $amatch = [];
        $pattern = '/^([\w[:punct:] ]+) ([0-9]{1,5})([\w[:punct:]\-\/]*)$/u';
        preg_match($pattern, $streetstr, $amatch);
        $street = (isset($amatch[1])) ? trim($amatch[1]) : '';
        $number = (isset($amatch[2])) ? trim($amatch[2]) : '';
        $numberAddition = (isset($amatch[3])) ? trim($amatch[3]) : '';
        
        if (empty($street) || empty($number)) {
            return [
                'validate' => false
            ];
        }
        return [
            'street' => $street,
            'number' => $number,
            'numberAddition' => str_replace("/", "", $numberAddition),
            'validate' => true
        ];
    }
}