<?php
/**
 * @author     polkurier.pl <api@polkurier.pl>
 * @copyright  polkurier.pl
 * @license    https://opensource.org/licenses/OSL-3.0 OSL/OSL-3.0
 * @category   Module
 * @package    Presta 1.7
 * @subpackage polkurier
 * @link       https://www.polkurier.pl
 */

namespace Polkurier\Model\Shipment\NewShipment;

use Polkurier\ConfigurationKey;
use Polkurier\Model\PolkurierConfig;
use Polkurier\Model\Shipment\CashonDelivery;
use Polkurier\Model\Shipment\ShipmentModel;
use Order;

class NewShipmentModel
{
    /**
     * @var ShipmentModel
     */
    private $shipmentmodel;

    /**
     * Id zamówienia w sklepie
     * @var int
     */
    private $id_order;

    public function __construct(ShipmentModel $shipmentmodel)
    {
        $this->shipmentmodel = $shipmentmodel;
    }

    /**
     * @param int $id_order
     * @return $this
     */
    public function setIdOrder($id_order)
    {
        $this->id_order = (int)$id_order;
        return $this;
    }


    /**
     * Dane dla nowego zamówienia
     * @throws \PrestaShopDatabaseException
     * @throws \PrestaShopException
     */
    public function NewOrder()
    {
        $this->shipmentmodel->setPrestaIdOrder($this->id_order);

        /* Adres odbiorcy */
        $this->shipmentmodel->setReceiver((new DeliveryAddressModel($this->id_order))->getDeliveryAddress());

        /* Adres nadawcy */
        $this->shipmentmodel->setSender((new SenderAddressModel())->getSenderAddress());

        /* Domyślny rodzaj przesyłki */
        $this->shipmentmodel->setShipmentType(PolkurierConfig::$default_shipment_type);

        /* Domyślny od banku */
        if (PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_BANK_ID)) {
            $this->shipmentmodel->getCodShipment()->setIdCod(PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_BANK_ID));
        }

        if (PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_RETURN_COD_TYPE)) {
            $this->shipmentmodel->getCodShipment()->setType(PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_RETURN_COD_TYPE));
        }

        $order = new Order($this->id_order);
        if ($this->isCOD($order) === true) {
            $codvalue = $this->getCodValue($order);
            $this->shipmentmodel->getCodShipment()->setValue($codvalue);
            $this->shipmentmodel->setInsuranceValue($codvalue);
            if (PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_RETURN_TIME_COD)) {
                $this->shipmentmodel->getCodShipment()->setTime(PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_RETURN_TIME_COD));
            }
        }
        $this->shipmentmodel->setOrderContent(sprintf('Zamówienie nr: %s', $order->reference));

        /* default carrier */
        if (PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_COURIER)) {
            $this->shipmentmodel->setServiceCode(PolkurierConfig::getConfigValue(ConfigurationKey::POLKURIER_DEFAULT_COURIER));
        }
    }

    /**
     * @param Order $order
     * @return bool
     */
    private function isCOD(Order $order)
    {
        return !(null === $order->module || !in_array($order->module, CashonDelivery::$cashondeliverymodule, true));
    }

    /**
     * @param Order $order
     * @return float
     */
    private function getCodValue(Order $order)
    {
        $total_paid_tax_incl = (float)number_format($order->total_paid_tax_incl, 2, '.','');
        if (in_array($order->module, CashonDelivery::$cashondeliverymodule, true)) {
            return $total_paid_tax_incl;
        }
        return (float)max(0, $total_paid_tax_incl - $this->getTotalPaid($order));
    }

    /**
     * @param Order $order
     * @return float
     */
    private function getTotalPaid(Order $order)
    {
        return (float)number_format($order->getTotalPaid(new \Currency(\Currency::getIdByIsoCode('PLN'))), 2, '.','');
    }
}