<?php
/**
 * @author     polkurier.pl <api@polkurier.pl>
 * @copyright  polkurier.pl
 * @license    https://opensource.org/licenses/OSL-3.0 OSL/OSL-3.0
 * @category   Module
 * @package    Presta 1.7
 * @subpackage polkurier
 * @link       https://www.polkurier.pl
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

if (!defined('_PS_MODULE_DIR_')) {
    echo '_PS_MODULE_DIR_ is not defined';
}

require_once(_PS_MODULE_DIR_ . "polkurier/autoload.php");

class polkurier extends CarrierModule
{
    const PRESTA_17 = '1.7';
    const PRESTA_16 = '1.6';
    private $prestaenv = self::PRESTA_17;
    private $pagecheckout = false;

    public function __construct()
    {
        $this->name = 'polkurier';
        $this->tab = 'shipping_logistics';
        $this->version = '1.1.30';
        $this->author = 'polkurier.pl';
        $this->need_instance = 0;
        $this->displayName = $this->l('POLKURIER.PL');
        $this->description = $this->l('Wysyłaj szybciej i wygodniej dzięki integracji z portalem Polkurier.pl');
        $this->confirmUninstall = $this->l('Czy na pewno chcesz odinstalować?');
        $this->ps_versions_compliancy = ['min' => '1.6.1.9', 'max' => _PS_VERSION_];

        parent::__construct();
        $this->bootstrap = true;
        $this->setEnvironment();
    }

    public function setEnvironment()
    {
        if (Tools::version_compare(_PS_VERSION_, '1.7', '<')) {
            $this->prestaenv = self::PRESTA_16;
        } else {
            $this->prestaenv = self::PRESTA_17;
        }

        if ($this->prestaenv === self::PRESTA_17 && Module::isEnabled('ets_onepagecheckout')) {
            $this->pagecheckout = true;
        }

        if ($this->prestaenv === self::PRESTA_16 && (Module::isEnabled('onepagecheckout') || (int)Configuration::get('PS_ORDER_PROCESS_TYPE') == 1)) {
            $this->pagecheckout = true;
        }
    }

    /**
     * @return bool
     */
    public function install()
    {
        if (extension_loaded('curl') === false) {
            $this->_errors[] = $this->l('You have to enable the cURL extension on your server to install this module');
            return false;
        }

        if (version_compare(PHP_VERSION, '5.6.0', '<')) {
            $this->_errors[] = 'Zalecana wersja PHP przynajmniej 5.6';
        }

        Sql\InstallSql::Install();

        if (!$this->installTab()) {
            return false;
        }

        if (!parent::install() || !$this->registerAdminHook() || !$this->registerFront()) {
            return false;
        }

        $this->registerConfigValue();

        $this->AddCarriers();

        return true;
    }


    private function registerConfigValue()
    {
        Polkurier\Model\PolkurierConfig::setConfigValue(Polkurier\ConfigurationKey::POLKURIER_CRON_TOKEN, self::uniqhash());
        Polkurier\Model\PolkurierConfig::setConfigValue(Polkurier\ConfigurationKey::POLKURIER_UPDATE_STATUS_TRAKING_DATA, date('Y-m-d H:i'));
        Polkurier\Model\PolkurierConfig::setConfigValue(Polkurier\ConfigurationKey::POLKURIER_INTERVAL_STATUS_CRON, ConfigApi::$interval_status_cron);
    }


    public function registerAdminHook()
    {
        $this->registerHook('displayAdminOrder');
        if (Tools::version_compare(_PS_VERSION_, '1.7.7', '<')) {
            $this->registerHook('displayAdminOrderContentShip');
        }
        return $this->registerHook('adminOrder') && $this->registerHook('backOfficeHeader') && $this->registerHook('DisplayBackOfficeFooter');
    }


    public function unregisterAdminHook()
    {
        $this->unregisterHook('displayAdminOrder');
        if (Tools::version_compare(_PS_VERSION_, '1.7.7', '<')) {
            $this->unregisterHook('displayAdminOrderContentShip');
        }
        $this->unregisterHook('adminOrder') && $this->unregisterHook('backOfficeHeader') && $this->unregisterHook('DisplayBackOfficeFooter');
    }

    /**
     * @return bool
     */
    private function registerFront()
    {
        return $this->registerHook('header') && $this->registerHook('displayCarrierExtraContent') && $this->registerHook('footer');
    }

    /**
     * @return bool
     */
    private function unregisterFront()
    {
        return $this->unregisterHook('header') && $this->unregisterHook('displayCarrierExtraContent') && $this->unregisterHook('footer');
    }



    /**
     * @return bool
     */
    public function uninstall()
    {
        $id_reference = self::getIdReferenceCarriers();
        foreach (self::getCarriers() as $item) {
            if ((int)$item['active'] === 1 && (int)$item['deleted'] === 0 && in_array((int)$item['id_reference'], $id_reference)) {
                self::UninstallCarrier((int)$item['id_carrier']);
            }
        }

        foreach (Polkurier\ConfigurationKey::$name as $key => $val) {
            Configuration::deleteByName($val);
        }

        Sql\UninstallSql::Uninstall();

        $this->unregisterAdminHook();

        $this->unregisterFront();

        $this->uninstallTab();

        return parent::uninstall();
    }


    /**
     * @return bool
     */
    public function installTab()
    {
        try {
            if (!(int)Tab::getIdFromClassName($this->name)) {
                $tab = new Tab();
                $tab->active = 1;
                $tab->class_name = $this->name;
                $tab->name = [];
                foreach (Language::getLanguages(true) as $lang) {
                    $tab->name[$lang['id_lang']] = $this->name;
                }
                $tab->module = $this->name;
                $tab->add();
            }

            if (!(int)Tab::getIdFromClassName('AdminPolkurierOrderModule')) {
                $tab = new Tab();
                $tab->active = 1;
                $tab->class_name = 'AdminPolkurierOrderModule';
                foreach (Language::getLanguages(true) as $lang) {
                    $tab->name[$lang['id_lang']] = 'Przesyłki POLKURIER.PL';
                }
                $tab->id_parent = (int)Tab::getIdFromClassName('AdminParentOrders');
                $tab->module = $this->name;
                $tab->add();
            }

            if (!(int)Tab::getIdFromClassName('AdminPolkurierModule')) {
                $tab = new Tab();
                $tab->active = 0;
                $tab->class_name = 'AdminPolkurierModule';
                foreach (Language::getLanguages(true) as $lang) {
                    $tab->name[$lang['id_lang']] = $this->name;
                }
                $tab->id_parent = Tab::getIdFromClassName($this->name);
                $tab->module = $this->name;
                $tab->add();
            }

        } catch (Exception $ex) {
            $this->_errors[] = $ex->getMessage();
            return false;
        }

        return true;
    }


    public function uninstallTab()
    {
        $tabarray = [
            $this->name,
            'AdminPolkurierOrderModule',
            'AdminPolkurierModule'
        ];

        foreach ($tabarray as $item) {
            $id_tab = (int)Tab::getIdFromClassName($item);
            if ($id_tab) {
                (new Tab($id_tab))->delete();
            }
        }
        return true;
    }


    public function AddCarriers()
    {
        $carriers_api_code = [];
        foreach (Polkurier\CarriersList::getCarriers() as $item) {
            $id_reference = $this->createCarriers($item);
            if ($id_reference) {
                $carriers_api_code[$id_reference] = ['api_code' => $item['api_code']];
            }
        }
        Polkurier\Model\PolkurierConfig::setConfigValue(Polkurier\ConfigurationKey::POLKURIER_CARRIERS_API_CODE, json_encode($carriers_api_code));
    }


    /**
     * @param array $param
     * @return false|int
     */
    public function createCarriers($param)
    {
        $carrier = new Carrier();
        $carrier->name = $this->l($param['name']);
        $carrier->active = 0;
        $carrier->is_free = 0;
        $carrier->shipping_handling = 1;
        $carrier->shipping_external = 0;
        $carrier->shipping_method = 1;
        $carrier->max_width = 0;
        $carrier->max_height = 0;
        $carrier->max_depth = 0;
        $carrier->max_weight = 0;
        $carrier->grade = 0;
        $carrier->is_module = 1;
        $carrier->need_range = 1;
        $carrier->range_behavior = 1;
        $carrier->external_module_name = $this->name;
        $carrier->url = $param['url'];

        $delay = [];
        foreach (Language::getLanguages(false) as $language) {
            $delay[$language['id_lang']] = $param['name'];
        }
        $carrier->delay = $delay;
        if (!$carrier->save()) {
            return false;
        }
        $groups = [];
        foreach (Group::getGroups((int)Context::getContext()->language->id) as $group) {
            $groups[] = $group['id_group'];
        }
        if (!$carrier->setGroups($groups)) {
            return false;
        }
        copy($this->local_path . 'views/img/' . $param['img'], _PS_SHIP_IMG_DIR_ . (int)$carrier->id . '.jpg');

        $carrier = new Carrier($carrier->id);
        return (int)$carrier->id_reference;
    }


    /**
     * @param int $carrier_id
     */
    private static function UninstallCarrier($carrier_id)
    {
        $carrier = new Carrier($carrier_id);
        if ((int)Configuration::get('PS_CARRIER_DEFAULT') === $carrier_id) {
            $carriers = Carrier::getCarriers(Context::getContext()->language->id, true, false, false, NULL, PS_CARRIERS_AND_CARRIER_MODULES_NEED_RANGE);
            foreach ($carriers as $item) {
                if ((int)$item['active'] === 1 && (int)$item['deleted'] === 0 && ((int)$item['id_carrier'] !== $carrier_id)) {
                    Configuration::updateValue('PS_CARRIER_DEFAULT', $item['id_carrier']);
                }
            }
        }
        $carrier->deleted = true;
        $carrier->save();
    }

    /**
     * @return array
     */
    private static function gethookCarriers()
    {
        return json_decode(Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_CARRIERS_API_CODE), true);
    }

    /**
     * @return array
     */
    private static function getIdReferenceCarriers()
    {
        $id_reference = [];
        foreach (self::gethookCarriers() as $key => $val) {
            $id_reference[] = (int)$key;
        }
        return $id_reference;
    }

    /**
     * @return bool
     */
    private static function isActiveCarriers()
    {
        $id_reference = self::getIdReferenceCarriers();
        $activ_carrier = false;
        foreach (self::getCarriers() as $item) {
            if ((int)$item['active'] === 1 && (int)$item['deleted'] === 0 && in_array((int)$item['id_reference'], $id_reference)) {
                $activ_carrier = true;
            }
        }
        return $activ_carrier;
    }

    /**
     * @return array
     */
    private static function getCarriers()
    {
        return Carrier::getCarriers(Context::getContext()->language->id, false, false, false, null, Carrier::ALL_CARRIERS);
    }

    /**
     * @param $params
     * @return mixed
     * @version  1.7
     */
    public function hookDisplayCarrierExtraContent($params)
    {
        $carriers_api_code = self::gethookCarriers();
        $id_reference = self::getIdReferenceCarriers();
        $point_api_code = [];
        foreach (self::getCarriers() as $item) {
            if (in_array((int)$item['id_reference'], $id_reference)) {
                $point_api_code[$item['id_carrier']] = $carriers_api_code[$item['id_reference']];
            }
        }
        $carrier_id = $params['carrier']['id'];

        if (!empty($point_api_code) && isset($point_api_code[$carrier_id])) {

            try {

                $carrier = $point_api_code[$carrier_id];
                $api_code = $carrier['api_code'];
                $chosen_collection_point = $searchQuery = '';

                $cart = $this->context->cart;
                $pointData = Polkurier\CollectionPoint\CollectionPointModel::getCollectionPointByIdCartProviders($this->context->cart->id, $api_code);
                if (!empty($pointData)) {
                    $chosen_collection_point = $pointData['collection_point'] . '|' . $pointData['adres'];
                } else {
                    $deliveryAddress = new Address($cart->id_address_delivery);
                    $searchQuery = $deliveryAddress->postcode . ' ' . $deliveryAddress->city;
                }

                $points_map_options = [
                    'providers' => $api_code,
                    'searchQuery' => $searchQuery,
                    'input_chosen' => 'input_chosen_point_' . $api_code,
                    'functions' => ['collect']
                ];

                $this->context->smarty->assign([
                    'polkurier_chosen_collection_point' => $chosen_collection_point,
                    'polkurier_points_map_options' => json_encode($points_map_options),
                    'polkurier_carrier' => $api_code
                ]);
                return $this->context->smarty->fetch($this->local_path . 'views/templates/front/collection_point_button.tpl');
            } catch (Exception $ex) {
                return '';
            }
        }
    }


    /**
     * Front
     * @return string
     * @return string
     * @throws Exception
     */
    public function hookHeader()
    {
        $controller = isset($this->context->controller) ? $this->context->controller : null;

        $allowed_controllers = array(
            'OrderController',
            'OrderOpcController',
            'OnepagecheckoutOrderModuleFrontController',
            'Ets_onepagecheckoutOrderModuleFrontController'
        );

        $is_allowed = !is_null($controller) && in_array(get_class($controller), $allowed_controllers);
        if (!$is_allowed) {
            return '';
        }

        if (self::isActiveCarriers() === false) {
            return '';
        }


        if ($this->prestaenv === self::PRESTA_17) {

            $this->context->controller->registerStylesheet(
                'polkurier-remote-points-map-css',
                ConfigApi::$maps_url_css,
                [
                    'server' => 'remote',
                    'position' => 'bottom',
                    'priority' => 20
                ]);

            $this->context->controller->registerJavascript(
                'polkurier-remote-points-map-js',
                ConfigApi::$maps_url_js,
                [
                    'priority' => 200,
                    'server' => 'remote',
                    'attribute' => 'async',
                ]);

            $this->context->controller->registerStylesheet(
                'polkurier-parcellocker-css',
                'modules/' . $this->name . '/views/css/polkurier_parcel_locker.css',
                [
                    'media' => 'all',
                    'priority' => 200,
                ]);

            $this->context->controller->registerJavascript(
                'polkurier-collection-point-widget-js',
                'modules/' . $this->name . '/views/js/collection_point_widget.js',
                [
                    'priority' => 200
                ]
            );

            if ($this->pagecheckout && get_class($controller) === 'Ets_onepagecheckoutOrderModuleFrontController') {

                $this->context->controller->registerJavascript(
                    'collection-point-one-page-checkout-17.js',
                    'modules/' . $this->name . '/views/js/collection_point_one_page_checkout_17.js',
                    [
                        'priority' => 200
                    ]
                );

            } else {
                $this->context->controller->registerJavascript(
                    'polkurier-collection-point-17-js',
                    'modules/' . $this->name . '/views/js/collection_point_17.js',
                    [
                        'priority' => 200
                    ]
                );
            }

        } else {

            $this->context->controller->addCSS(ConfigApi::$maps_url_css);
            $this->context->controller->addJS($this->_path . '/views/js/collection_point_widget.js');
            $this->context->controller->addCSS($this->_path . '/views/css/polkurier_parcel_locker_16.css');

            if ($this->pagecheckout) {
                if (in_array(get_class($controller), ['OnepagecheckoutOrderModuleFrontController', 'OrderOpcController'])) {
                    $this->context->controller->addJS($this->_path . '/views/js/collection_point_one_page_checkout_16.js');
                } else {
                    $this->context->controller->addJS($this->_path . '/views/js/collection_point_16.js');
                }
            } else {
                $this->context->controller->addJS($this->_path . '/views/js/collection_point_16.js');
            }
        }
        return $this->hookHeaderCarrier();
    }


    public function  hookFooter()
    {
        return '<script src="'.ConfigApi::$maps_url_js.'"></script>';
    }

    /**
     * @return string
     * @throws Exception
     */
    public function hookHeaderCarrier()
    {
        $cart = $this->context->cart;
        $AjaxUrl = $this->context->link->getModuleLink($this->name, 'CollectionPoint', ['ajax' => true]);
        $AjaxDeleteUrl = $this->context->link->getModuleLink($this->name, 'DeleteCollectionPoint', ['ajax' => true]);

        $polkurier_map_token = '';
        try {
            $polkurier_map_token = (new Polkurier\ApiMetod\GetMapTokenApi())->getToken();
        } catch (Exception $ex) {
        }
        $polkurier_maps_url = ConfigApi::$maps_url;
        $polkurier_collection_point_token = Tools::encrypt($cart->id);

        $hook_carriers_code = self::gethookCarriers();
        $id_reference = self::getIdReferenceCarriers();
        $point_api_code = [];

        foreach (self::getCarriers() as $item) {
            if ((int)$item['active'] === 1 && (int)$item['deleted'] === 0 && in_array((int)$item['id_reference'], $id_reference)) {
                $point_api_code[$item['id_carrier']] = $hook_carriers_code[$item['id_reference']]['api_code'];
            }
        }

        if ($this->prestaenv === self::PRESTA_17) {

            Media::addJsDef([
                'ID_CART' => $cart->id,
                'POLKURIER_COLLECTION_POINT_AJAX_URL' => $AjaxUrl,
                'POLKURIER_DELETE_COLLECTION_POINT_AJAX_URL' => $AjaxDeleteUrl,
                'POLKURIER_COLLECTION_POINT_TOKEN' => $polkurier_collection_point_token,
                'POLKURIER_COLLECTION_POINT_API_CODE' => $point_api_code,
                'POLKURIER_MAP_TOKEN' => $polkurier_map_token,
                'POLKURIER_MAPS_URL' => $polkurier_maps_url,
            ]);

        } else {

            $cart = $this->context->cart;
            $chosen_collection_point = [];
            $searchQuery = '';
            foreach ($hook_carriers_code as $item) {
                $pointData = Polkurier\CollectionPoint\CollectionPointModel::getCollectionPointByIdCartProviders($cart->id, $item['api_code']);
                if (!empty($pointData)) {
                    $chosen_collection_point[$item['api_code']] = $pointData['collection_point'] . '|' . $pointData['adres'];
                } else if (empty($searchQuery)) {
                    $deliveryAddress = new Address($cart->id_address_delivery);
                    $searchQuery = $deliveryAddress->postcode . ' ' . $deliveryAddress->city;
                }
            }

            $this->context->smarty->assign(
                [
                    'ID_CART' => $cart->id,
                    'POLKURIER_COLLECTION_POINT_AJAX_URL' => $AjaxUrl,
                    'POLKURIER_DELETE_COLLECTION_POINT_AJAX_URL' => $AjaxDeleteUrl,
                    'POLKURIER_COLLECTION_POINT_TOKEN' => $polkurier_collection_point_token,
                    'POLKURIER_COLLECTION_POINT_API_CODE' => $point_api_code,
                    'POLKURIER_MAP_TOKEN' => $polkurier_map_token,
                    'POLKURIER_MAPS_URL' => $polkurier_maps_url,
                    'POLKURIER_CHOSEN_COLLECTION_POINT' => $chosen_collection_point,
                    'POLKURIER_SEARCH_QUERY' => $searchQuery
                ]
            );
            return $this->display(__FILE__, 'views/templates/front/collection_point_front_header.tpl');
        }
        return '';
    }

    /**
     * TAB wysyłka
     * @param $params
     * @return mixed
     */
    public function hookDisplayAdminOrderContentShip($params)
    {
        return $this->ListOrders($params);
    }


    /**
     * @param $params
     * @return mixed
     */
    private function ListOrders($params)
    {
        if (isset($params['order']) && $params['order'] instanceof Order) {
            $order = $params['order'];
        } else {
            $order = new Order($params['id_order']);
        }

        $order_polkurier = (new Polkurier\Order\OrderPolkurier())->ListOrdersByPrestaId((int)$order->id_cart);
        $this->context->smarty->assign([
            'orders' => $order_polkurier
        ]);
        return $this->display(__FILE__, 'views/templates/hook/orde_list.tpl');
    }

    /**
     * @param $params
     * @param $shipping_cost
     * @return float
     */
    public function getOrderShippingCost($params, $shipping_cost)
    {
        return 0.0;
    }

    /**
     * @param $params
     * @return float
     */
    public function getOrderShippingCostExternal($params)
    {
        return 0.0;
    }


    private function LoadConfig()
    {
        Polkurier\Model\PolkurierConfig::init();
    }


    /**
     * generating an array of driveway dates courier
     * @return array
     */
    private static function getDatePickupArray()
    {
        $counter = 0;
        $countday = 0;
        $dataout = [];
        do {
            $date = (new \DateTime())->modify("+" . $countday . "day");
            if (((int)$date->format("N") !== 6) && ((int)$date->format("N") !== 7)) {
                $dataout[] = $date->format('Y-m-d');
                $counter++;
            }
            $countday++;
        } while ($counter < 6);

        return $dataout;
    }


    /**
     * @throws Exception
     */
    public function hookBackOfficeHeader()
    {
        if (Tools::getValue('controller') === 'AdminOrders') {

            $this->context->controller->addJquery();
            $this->context->controller->addCSS([$this->_path . '/views/select2/css/select2.min.css']);
            $this->context->controller->addJS([$this->_path . '/views/select2/js/select2.min.js']);

            $AjaxUrl = $this->context->link->getModuleLink($this->name, 'CollectionPoint', ['ajax' => true]);

            $maptokenapi = '';
            try {
                $maptokenapi = (new Polkurier\ApiMetod\GetMapTokenApi())->getToken();
            } catch (Exception $ex) {
            }
            Media::addJsDef([
                'POLKURIER_MAPS_URL' => ConfigApi::$maps_url,
                'POLKURIER_MAP_TOKEN' => $maptokenapi,
                'POLKURIER_COLLECTION_POINT_AJAX_URL' => $AjaxUrl
            ]);

            $this->context->controller->addCSS([
                ConfigApi::$maps_url_css
            ]);

            $this->context->controller->addJS([
                $this->_path . '/views/js/collection_point_widget_admin.js',
                $this->_path . '/views/js/collection_point_admin.js'
            ]);
        }
    }

    public function hookDisplayBackOfficeFooter()
    {
        return '<script src="'.ConfigApi::$maps_url_js.'"></script>';
    }


    /**
     * order form
     * @param array $params
     * @return string
     * @throws Exception
     */
    public function hookdisplayAdminOrder($params)
    {
        $this->LoadConfig();

        $shipmentModel = new Polkurier\Model\Shipment\ShipmentModel();
        $NewShipment = new Polkurier\Model\Shipment\NewShipment\NewShipmentModel($shipmentModel);
        $NewShipment->setIdOrder($params['id_order']);
        $NewShipment->NewOrder();
        $show_order_form = true;

        $msg = new Polkurier\Msg();

        try {

            Polkurier\Model\PolkurierConfig::CheckIsDefined();

            $apiclient = new Polkurier\ApiMetod\ApiClient();
            $apiclient->checkAPI();

        } catch (Exception $e) {

            $msg->setErrors($e->getMessage());
            $show_order_form = false;
        }

        $default_pickup_type = Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_PICKUP_TYPE);
        $variables = [
            'ps_base_uri' => __PS_BASE_URI__,
            'updatemodule' => $this->getUpdateModule(),
            'date_pickup_array' => self::getDatePickupArray(),
            'shipmentmodel' => $shipmentModel,
            'parcel_template_list' => Polkurier\Model\PolkurierConfig::$parcel_template_list->getListOption(),
            'parcel_template_list_json' => Polkurier\Model\PolkurierConfig::$parcel_template_list->getListToJson(),
            'sender_list' => Polkurier\Model\PolkurierConfig::$sender_list->getList(),
            'sender_list_json' => Polkurier\Model\PolkurierConfig::$sender_list->getListToJson(),
            'default_pickup_type' => $default_pickup_type ? $default_pickup_type : Polkurier\Model\PolkurierConfig::$default_pickup_type,
            'POLKURIER_DEFAULT_SENDER_ID' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_SENDER_ID),
            'POLKURIER_TEMPLATE_PARCEL_ID' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_TEMPLATE_PARCEL_ID),
            'countries' => Polkurier\Model\PolkurierConfig::$countries->getListOption(),
            'codtype' => Polkurier\Model\PolkurierConfig::$codtype,
            'bank_list' => Polkurier\Model\PolkurierConfig::$bank_list->getList(),
            'cod_return_time_default' => Polkurier\Model\PolkurierConfig::$cod_return_time_default,
            'cod_return_time' => Polkurier\Model\PolkurierConfig::$cod_return_time,
            'shipment_type' => Polkurier\Model\PolkurierConfig::$shipment_type,
            'couriers' => Polkurier\Model\PolkurierConfig::$couriers,
            'show_order_form' => $show_order_form,
            'polkurier_token' => Tools::getAdminTokenLite('AdminPolkurierModule'),
            'polkurier_error_msg' => $this->getDisplayErrorHtml($msg),
            'polkurier_confirm_msg' => $this->getDisplayConfirmationHtml($msg),
            'polkurier_sender_point_inpost_paczkomat' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_INPOST_PACZKOMAT),
            'polkurier_sender_point_pocztex_punkt' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_POCZTEX),
            'polkurier_order_sms_notification' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_SMS_NOTIFICATION),
            'polkurier_sender_point_dpd' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_DPD),
            'polkurier_sender_point_ups' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_UPS),
        ];
        $this->context->smarty->assign($variables);
        return $this->display(__FILE__, OrderView::getTemplate());
    }


    /**
     * @param Polkurier\Msg $msg
     * @return array
     */
    public function getDisplayErrorHtml(Polkurier\Msg $msg)
    {
        $messages = [];
        foreach ($msg->getErrors() as $error) {
            if (!empty($error)) {
                $messages[] = $this->displayError($this->l($error));
            }
        }
        return $messages;
    }

    /**
     * @param Polkurier\Msg $msg
     * @return array
     */
    public function getDisplayConfirmationHtml(Polkurier\Msg $msg)
    {
        $messages = [];
        foreach ($msg->getConfirmations() as $confirmation) {
            if (!empty($confirmation)) {
                $messages[] = $this->displayConfirmation($this->l($confirmation));
            }
        }
        return $messages;
    }


    /**
     * @return string
     * @throws ReflectionException|Exception
     */
    public function getContent()
    {
        $this->LoadConfig();

        $this->context->smarty->assign([
            'module_dir' => $this->_path,
            'cron_token' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_CRON_TOKEN)
        ]);

        if (Tools::getValue('section') === 'auth') {
            return $this->AuthDataControl();
        }

        if (Tools::getValue('section') === 'settemplate') {
            return $this->TemplateControl();
        }

        if (Tools::getValue('section') === 'status_order') {
            return $this->StatusOrderControl();
        }

        if (Tools::getValue('section') === 'sender_point') {
            return $this->SenderPointControl();
        }

        if (Tools::getValue('section') === 'template_sender') {
            return $this->SenderTemplateControl();
        }

        if (Tools::getValue('section') === 'template_parcel') {
            return $this->ParcelTemplateControl();
        }

        if (Tools::getValue('section') === 'template_bank') {
            return $this->BankTemplateControl();
        }

        return $this->showAdminConfig();
    }

    /**
     * @return string
     */
    private function showAdminConfig()
    {

        $this->msg = new Polkurier\Msg();

        $this->context->controller->addJquery();
        $this->context->controller->addCSS([$this->_path . '/views/select2/css/select2.min.css']);
        $this->context->controller->addJS([$this->_path . '/views/select2/js/select2.min.js']);

        $varupdatemodule['updatemodule'] = $this->getUpdateModule();
        $varupdatemodule['version'] = $this->version;
        $this->context->smarty->assign($varupdatemodule);

        /* nagłówek */
        $head = $this->context->smarty->fetch($this->local_path . AdminConfigTopView::getTemplate());

        /* dane autoryzacyjne */
        $output = $this->renderAuthForm();

        /* domyślne dane */
        $output .= $this->renderTemplateForm();

        /* domyślne punkty nadawcze */
        $output .= $this->renderSenderPointForm();

        /* Zmiana statusów */
        $output .= $this->renderStatusOrderForm();

        $variables['sender_list'] = Polkurier\Model\PolkurierConfig::$sender_list;
        $variables['parcel_template_list'] = Polkurier\Model\PolkurierConfig::$parcel_template_list;
        $variables['bank_list'] = Polkurier\Model\PolkurierConfig::$bank_list;
        $variables['module_dir'] = $this->_path;
        $variables['base_url'] = $this->context->link->getAdminLink('AdminModules') . '&configure=' . $this->name;
        $variables['polkurier_token'] = Tools::getAdminTokenLite('AdminPolkurierModule');
        $variables['polkurier_sender_point_inpost_paczkomat'] = Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_INPOST_PACZKOMAT);
        $variables['polkurier_sender_point_pocztex_punkt'] = Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_POCZTEX);
        $variables['polkurier_sender_point_dpd'] = Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_DPD);
        $variables['polkurier_sender_point_ups'] = Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_UPS);
        $this->context->smarty->assign($variables);

        $infohtml = '';
        $confirmation = $this->getDisplayConfirmationHtml($this->msg);
        if (is_array($confirmation)) {
            foreach ($confirmation as $item) {
                $infohtml .= $item;
            }
        }

        $error = $this->getDisplayErrorHtml($this->msg);
        if (is_array($error)) {
            foreach ($error as $item) {
                $infohtml .= $item;
            }
        }

        return $head . $infohtml . $output . $this->display(__FILE__, AdminConfigView::getTemplate());
    }


    private function redirectToAdminConfig()
    {
        Tools::redirectAdmin(Context::getContext()->link->getAdminLink('AdminModules') . '&configure=' . $this->name);
    }

    /**
     * @return bool|string
     */
    private static function getAdminModulesToken()
    {
        return Tools::getAdminTokenLite('AdminModules');
    }


    /**
     * authorization data
     * @return void
     */
    private function AuthDataControl()
    {
        if (Tools::getValue('action') === 'save') {

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_AUTH_API_ACCOUNT,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_AUTH_API_ACCOUNT, '')
            );
            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_AUTH_API_TOKEN,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_AUTH_API_TOKEN, '')
            );
        }
        $this->redirectToAdminConfig();
    }

    /**
     * @return string
     * @throws PrestaShopException
     */
    private function renderAuthForm()
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&section=auth&action=save';
        $helper->token = self::getAdminModulesToken();

        $helper->tpl_vars = array(
            'fields_value' => [
                'POLKURIER_AUTH_API_ACCOUNT' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_AUTH_API_ACCOUNT),
                'POLKURIER_AUTH_API_TOKEN' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_AUTH_API_TOKEN),
            ],
        );

        try {

            Polkurier\Model\PolkurierConfig::CheckIsDefined();
            $apiclient = new Polkurier\ApiMetod\ApiClient();
            $apiclient->checkAPI();

        } catch (Exception $ex) {
            $this->msg->setErrors($ex->getMessage());
        }
        return $helper->generateForm(array(ConfigAuthView::getFormConfig()));
    }


    /**
     * default settings
     * @return void
     */
    private function TemplateControl()
    {
        if (Tools::getValue('action') === 'save') {

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_DEFAULT_COURIER,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_COURIER, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_DEFAULT_PICKUP_TYPE,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_PICKUP_TYPE, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_TEMPLATE_PARCEL_ID,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_TEMPLATE_PARCEL_ID, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_DEFAULT_SENDER_ID,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_SENDER_ID, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_DEFAULT_BANK_ID,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_BANK_ID, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_DEFAULT_RETURN_TIME_COD,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_RETURN_TIME_COD, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_SMS_NOTIFICATION,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_SMS_NOTIFICATION, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_RETURN_COD_TYPE,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_RETURN_COD_TYPE, '')
            );

        }

        $this->redirectToAdminConfig();
    }

    /**
     * @return string
     */
    private function renderTemplateForm()
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&section=settemplate&action=save';
        $helper->token = self::getAdminModulesToken();

        $helper->tpl_vars = [
            'fields_value' => [
                'POLKURIER_DEFAULT_COURIER' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_COURIER),
                'POLKURIER_DEFAULT_PICKUP_TYPE' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_PICKUP_TYPE),
                'POLKURIER_TEMPLATE_PARCEL_ID' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_TEMPLATE_PARCEL_ID),
                'POLKURIER_DEFAULT_SENDER_ID' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_SENDER_ID),
                'POLKURIER_DEFAULT_BANK_ID' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_BANK_ID),
                'POLKURIER_SET_ORDER_SMS_NOTIFICATION' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_SMS_NOTIFICATION),
                'POLKURIER_DEFAULT_RETURN_TIME_COD' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_DEFAULT_RETURN_TIME_COD),
                'POLKURIER_RETURN_COD_TYPE' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_RETURN_COD_TYPE)
            ],
        ];

        ConfigTemplateView::setVariables(
            'couriers',
            array_merge(['...' => ['id' => '', 'name' => '...']], Polkurier\Model\PolkurierConfig::$couriers)
        );
        ConfigTemplateView::setVariables(
            'parcel_list',
            array_merge(['...' => ['id' => '', 'name' => '...']], Polkurier\Model\PolkurierConfig::$parcel_template_list->getListOption())
        );
        ConfigTemplateView::setVariables(
            'sender_list',
            array_merge(['...' => ['id' => '', 'name' => '...']], Polkurier\Model\PolkurierConfig::$sender_list->getListOption())
        );
        ConfigTemplateView::setVariables(
            'template_bank',
            array_merge(['...' => ['id' => '', 'name' => '...']], Polkurier\Model\PolkurierConfig::$bank_list->getListOption())
        );

        foreach (Polkurier\CodReturnTime::NAME as $key => $item) {
            $return_time_cod[] = ['id' => $key, 'name' => $item];
        }
        ConfigTemplateView::setVariables('return_time_cod', $return_time_cod);

        $return_cod_type = [];
        foreach (Polkurier\CodType::NAME as $key => $item) {
            $return_cod_type[] = ['id' => $key, 'name' => $item];
        }
        ConfigTemplateView::setVariables('return_cod_type', $return_cod_type);


        $output = $this->context->smarty->fetch($this->local_path . ConfigTemplateView::getTemplate());

        return $output . $helper->generateForm([ConfigTemplateView::getFormConfig()]);
    }


    /**
     * Default sender address
     * @return string
     */
    private function SenderTemplateControl()
    {
        switch (Tools::getValue('action')) {
            case 'save':
                Polkurier\Sender\Sender::SaveSenderData(Tools::getAllValues());
                break;
            case 'edit':
                return $this->renderSenderTemplateForm(Tools::getValue('id'));
                break;
            case 'deletesender':
                if (is_numeric(Tools::getValue('id'))) {
                    Polkurier\Sender\SenderModel::deleteSender(Tools::getValue('id'));
                }
                break;
            default:
                return $this->renderSenderTemplateForm();
                break;
        }
        $this->redirectToAdminConfig();
    }


    /**
     * @param string $id
     * @return string
     */
    private function renderSenderTemplateForm($id = '')
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&section=template_sender&action=save';
        $helper->token = self::getAdminModulesToken();
        $helper->show_cancel_button = true;
        $senderfield = [
            'sender_id' => '',
            'name' => '',
            'company' => '',
            'first_name' => '',
            'last_name' => '',
            'street' => '',
            'house_no' => '',
            'flat_no' => '',
            'postcode' => '',
            'city' => '',
            'contact_name' => '',
            'email' => '',
            'phone' => '',
        ];

        if (is_numeric($id)) {
            $senderfield = array_merge($senderfield, Polkurier\Sender\SenderModel::getSenderById($id));
        }
        $helper->tpl_vars = [
            'fields_value' => $senderfield
        ];

        $output = $this->context->smarty->fetch($this->local_path . ConfigSenderView::getTemplate());

        return $output . $helper->generateForm([ConfigSenderView::getFormConfig()]);
    }


    /**
     * templates packages
     * @return string
     * @throws Exception
     */
    private function ParcelTemplateControl()
    {
        switch (Tools::getValue('action')) {
            case 'save':
                Polkurier\Parcel\Parcel::SaveParcelData(Tools::getAllValues());
                break;
            case 'edit':
                return $this->renderParcelTemplateForm(Tools::getValue('id'));
                break;
            case 'delete_parcel':
                if (is_numeric(Tools::getValue('id'))) {
                    Polkurier\Parcel\ParcelModel::deleteParcel(Tools::getValue('id'));
                }
                break;
            default:
                return $this->renderParcelTemplateForm();
                break;
        }
        $this->redirectToAdminConfig();
    }


    /**
     * @param string $id
     * @return string
     */
    private function renderParcelTemplateForm($id = '')
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&section=template_parcel&action=save';
        $helper->token = self::getAdminModulesToken();
        $helper->show_cancel_button = true;

        $parcel_field = [
            'parcel_id' => '',
            'name' => '',
            'shipment_type' => '',
            'nstd' => '',
            'content' => '',
            'quantity' => '',
            'weight' => '',
            'dim_1' => '',
            'dim_2' => '',
            'dim_3' => ''
        ];

        if (is_numeric($id)) {
            $parcel_field = array_merge($parcel_field, Polkurier\Parcel\ParcelModel::getParcelById($id));
        }
        $helper->tpl_vars = [
            'fields_value' => $parcel_field
        ];

        ConfigParcelView::setVariables('shipment_type', Polkurier\Model\PolkurierConfig::$shipment_type);
        $output = $this->context->smarty->fetch($this->local_path . ConfigParcelView::getTemplate());
        return $output . $helper->generateForm([ConfigParcelView::getFormConfig()]);
    }

    /**
     * bank account numbers
     * @return string
     */
    private function BankTemplateControl()
    {
        switch (Tools::getValue('action')) {
            case 'save':
                Polkurier\Bank\Bank::SaveBankAccountData(Tools::getAllValues());
                break;
            case 'edit':
                return $this->renderBankTemplateForm(Tools::getValue('id'));
                break;
            case 'delete_bank':
                if (is_numeric(Tools::getValue('id'))) {
                    Polkurier\Bank\BankModel::deleteBankAccount(Tools::getValue('id'));
                }
                break;
            default:
                return $this->renderBankTemplateForm();
                break;
        }
        $this->redirectToAdminConfig();
    }

    /**
     * @param string $id
     * @return string
     */
    private function renderBankTemplateForm($id = '')
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&section=template_bank&action=save';
        $helper->token = self::getAdminModulesToken();
        $helper->show_cancel_button = true;

        $bank_field = [
            'bank_id' => '',
            'name' => '',
            'iban' => '',
        ];

        if (is_numeric($id)) {
            $bank_field = array_merge($bank_field, Polkurier\Bank\BankModel::getBankById($id));
        }
        $helper->tpl_vars = [
            'fields_value' => $bank_field
        ];

        $output = $this->context->smarty->fetch($this->local_path . ConfigBankAccountView::getTemplate());
        return $output . $helper->generateForm([ConfigBankAccountView::getFormConfig()]);
    }


    /**
     * @return void
     */
    private function StatusOrderControl()
    {
        if (Tools::getValue('action') === 'save') {

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_NEW_SHIPMENT,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_NEW_SHIPMENT, ''
                )
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_STATUS_ORDER_NEW_SHIPMENT,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_STATUS_ORDER_NEW_SHIPMENT, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_STATUS_ORDER_STATUS_SHIPMENT,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_STATUS_ORDER_STATUS_SHIPMENT, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_STATUS_SHIPMENT,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_STATUS_SHIPMENT, '')
            );
        }

        $this->redirectToAdminConfig();
    }

    /**
     * @return array
     */
    public function getOrderStatus()
    {
        $statuses = OrderState::getOrderStates(Context::getContext()->language->id);
        $status = [
            [
                'id_order_state' => -1,
                'name' => '--'
            ]
        ];
        return array_merge($status, $statuses);
    }

    /**
     * @return string
     */
    private function renderStatusOrderForm()
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&section=status_order&action=save';
        $helper->token = self::getAdminModulesToken();

        $helper->tpl_vars = array(
            'fields_value' => [
                'POLKURIER_SET_ORDER_NEW_SHIPMENT' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_NEW_SHIPMENT),
                'POLKURIER_STATUS_ORDER_NEW_SHIPMENT' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_STATUS_ORDER_NEW_SHIPMENT),
                'POLKURIER_STATUS_ORDER_STATUS_SHIPMENT' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_STATUS_ORDER_STATUS_SHIPMENT),
                'POLKURIER_SET_ORDER_STATUS_SHIPMENT' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SET_ORDER_STATUS_SHIPMENT)
            ],
        );
        ConfigStatusOrderView::setVariables('order_status', $this->getOrderStatus());
        return $helper->generateForm([ConfigStatusOrderView::getFormConfig()]);
    }


    /**
     * @return void
     */
    private function SenderPointControl()
    {
        if (Tools::getValue('action') === 'save') {
            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_INPOST_PACZKOMAT,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_INPOST_PACZKOMAT, ''
                )
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_POCZTEX,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_POCZTEX, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_DPD,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_DPD, '')
            );

            Polkurier\Model\PolkurierConfig::setConfigValue(
                Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_UPS,
                Tools::getValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_UPS, '')
            );
        }
        $this->redirectToAdminConfig();
    }


    /**
     * @return string
     */
    private function renderSenderPointForm()
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&section=sender_point&action=save';
        $helper->token = self::getAdminModulesToken();
        $helper->tpl_vars = array(
            'fields_value' => [
                'POLKURIER_SENDER_POINT_INPOST_PACZKOMAT' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_INPOST_PACZKOMAT),
                'POLKURIER_SENDER_POINT_KURIER48' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_POCZTEX),
                'POLKURIER_SENDER_POINT_DPD' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_DPD),
                'POLKURIER_SENDER_POINT_UPS' => Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_SENDER_POINT_UPS),
            ],
        );
        return $helper->generateForm([ConfigSenderPointView::getFormConfig()]);
    }


    public function CronOrderStatus()
    {
        $cron_token = Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_CRON_TOKEN);
        $token = Tools::getValue('token');

        if (!empty($cron_token) && !empty($token) && $cron_token === $token) {

            if ((time() - strtotime(Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_UPDATE_STATUS_TRAKING_DATA))) >= (int)Polkurier\Model\PolkurierConfig::getConfigValue(Polkurier\ConfigurationKey::POLKURIER_INTERVAL_STATUS_CRON)) {

                (new Polkurier\Order\RefreshOrderStatusPolkurier())->Refresh();

                Polkurier\Model\PolkurierConfig::setconfigvalue('polkurier_update_status_traking_data', date('Y-m-d H:i'));
            }
        }
    }

    /**
     * @return string
     */
    public static function uniqhash()
    {
        return md5(uniqid(microtime(true), true));
    }

    /**
     * @return array
     */
    private function getUpdateModule()
    {
        $output = [
            'newversion' => false,
            'newupdate' => false
        ];
        try {
            $file = Tools::file_get_contents(ConfigApi::$url_updatemodule, true, null, 10);
            if ($file) {
                $json = json_decode($file, true);
                if (Tools::version_compare($this->version, $json['version'], '<') === true) {
                    $output['newversion'] = true;
                    $output['version'] = $json['version'];
                    $output['description'] = $json['description'];
                    $output['version_date'] = $json['version_date'];
                }
            }
        } catch (Exception $exp) {
        }
        return $output;
    }
}
