import {html, css, LitElement} from "../esm/libs/lit.dist.js?ver=2.5.1";
import globalStyles from '../esm/custom-elements/styles.css.js?ver=2.5.1';
import styles from './pk-quick-order-view.css.js?ver=2.5.1';
import {ProviderController} from "../esm/lit-controllers/provider-controller.js?ver=2.5.1";
import {CouriersProvider} from "../esm/providers/couriers-provider.js?ver=2.5.1";
import '../esm/custom-elements/pk-grid.js?ver=2.5.1';
import '../esm/custom-elements/pk-card.js?ver=2.5.1';
import '../esm/custom-elements/pk-icon.js?ver=2.5.1';
import {formatCurrency} from "../esm/util/numbers.js?ver=2.5.1";


export class PkQuickCourierView extends LitElement {

    static styles = [
        globalStyles,
        styles,
        css`
            pk-card {
                cursor: pointer;
                transition: all 0.2s;
                height: 100%;
                
                &:hover {
                    scale: 1.1;
                    z-index: 90;
                    transition: all 0.2s;
                    box-shadow: var(--pk-box-shadow-raised);
                }
                
                &.active {
                    --pk-card-background: var(--pk-blue-600);
                    --pk-card-border-color: var(--pk-blue-800);
                    --pk-icon-bg: rgba(255,255,255,0.05);
                    --pk-icon-color: white;
                    color: white;
                }
            }
        `
    ];

    static properties = {
        order: {},
        orderEvaluation: {},
        isEvaluationRequested: {type: Boolean},
    }

    couriersProvider = ProviderController.create(this, new CouriersProvider());

    getCourierEvaluation(courier) {
        if (courier && this.orderEvaluation) {
            for (let evaluation of this.orderEvaluation) {
                if (evaluation.servicecode === courier) {
                    return evaluation;
                }
            }
        }
    }

    render() {
        const validCouriers = this.couriersProvider.all().filter(courier => this.getCourierEvaluation(courier.servicecode));
        return html`
            ${validCouriers.length === 0  && !this.isEvaluationRequested ? html`
                <div class="d-inline-flex gap-sm text-danger">
                    <pk-icon type="plain" size="text" icon="exclamation-triangle"></pk-icon>
                    <div>Niestety, nie możemy wyszukać oferty do wybranych usług dodatkowych. Zmień parametry i sprawdź ponownie</div>
                </div>
            ` : null}
            ${validCouriers.length === 0  && this.isEvaluationRequested ? html`
                <div class="text-center">
                    <pk-spinner></pk-spinner>
                </div>
            ` : null}
            <pk-grid style="--gap-x: 0.5rem; --gap-y: 0.5rem;">
                ${validCouriers.map(courier => {
                    const evaluation = this.getCourierEvaluation(courier.servicecode);
                    return html`
                        <pk-col style="--size-md: 6; --size-lg: 4;">
                            <pk-card 
                                @click="${() => {
                                    if (evaluation) {
                                        this.dispatchEvent(new CustomEvent('select', {detail: courier}));
                                    }
                                }}"
                                class="${!evaluation ? 'inactive' : ''} ${courier.servicecode === this.order.courier ? 'active' : ''}"
                            >
                                <div class="d-flex gap">
                                    <div>
                                        <pk-icon icon="truck-fast" size="sm"></pk-icon>
                                    </div>
                                    <div>
                                        <div class="fs-sm fw-bold text-nowrap_">
                                            ${courier.name}
                                        </div>
                                        <div class="fs-nano">
                                            <strong>${evaluation?.netprice ? formatCurrency(evaluation.netprice) : '–'}</strong> netto
                                        </div>
                                        <div class="fs-nano">
                                            <strong>${evaluation?.grossprice ? formatCurrency(evaluation.grossprice) : '–'}</strong> brutto
                                        </div>
                                    </div>
                                </div>
                            </pk-card>
                        </pk-col>
                    `
                })}
            </pk-grid>
        `;
    }
}

customElements.define('pk-quick-courier-view', PkQuickCourierView);
