import {html, unsafeHTML, LitElement} from "../esm/libs/lit.dist.js?ver=2.5.1";
import {
    closeParentDialog,
    MODAL_SIZE_DEFAULT,
    MODAL_SIZE_EXTRA_LARGE,
    MODAL_SIZE_MEDIUM,
    MODAL_SIZE_SMALL
} from "../esm/custom-elements/pk-dialog.js?ver=2.5.1";
import {showErrorPopup} from "../esm/util/errors.js?ver=2.5.1";
import {ajaxRequest} from "../esm/util/http.js?ver=2.5.1";
import globalStyles from '../esm/custom-elements/styles.css.js?ver=2.5.1';
import styles from './pk-quick-order-view.css.js?ver=2.5.1';
import {TemplatesProvider} from "../esm/providers/templates-provider.js?ver=2.5.1";
import {ProviderController} from "../esm/lit-controllers/provider-controller.js?ver=2.5.1";
import {SenderAddressesProvider} from "../esm/providers/sender-addresses-provider.js?ver=2.5.1";
import {CouriersProvider} from "../esm/providers/couriers-provider.js?ver=2.5.1";
import {BankAccountsProvider} from "../esm/providers/bank-accounts-provider.js?ver=2.5.1";
import {CourierMessagesProvider} from "../esm/providers/courier-messages-provider.js?ver=2.5.1";
import {getBaseCourier, isSenderPointRequired} from "../esm/util/couriers.js?ver=2.5.1";
import {debounce} from "../esm/util/browser.js?ver=2.5.1";
import {createConfirm} from '../esm/custom-elements/pk-alert-dialog.js?ver=2.5.1';
import {createToast, TOAST_TYPE_SUCCESS} from "../esm/custom-elements/pk-toast-message.js?ver=2.5.1";
import {formatCurrency, numberFilter} from "../esm/util/numbers.js?ver=2.5.1";
import '../esm/custom-elements/pk-grid.js?ver=2.5.1';
import '../esm/custom-elements/pk-badge.js?ver=2.5.1';
import '../esm/custom-elements/pk-alert.js?ver=2.5.1';
import '../esm/custom-elements/pk-card.js?ver=2.5.1';
import '../esm/custom-elements/pk-icon.js?ver=2.5.1';
import '../esm/custom-elements/pk-loading-view.js?ver=2.5.1';
import './pk-quick-order-templates.js?ver=2.5.1';
import './pk-order-pickup-view.js?ver=2.5.1';
import './pk-address-view.js?ver=2.5.1';
import './pk-quick-order-pack-view.js?ver=2.5.1';
import './pk-quick-created-order-view.js?ver=2.5.1';
import './pk-quick-courier-view.js?ver=2.5.1';
import './pk-quick-order-address-select-view.js?ver=2.5.1';
import './pk-quick-order-recipient-edit-view.js?ver=2.5.1';
import './pk-quick-order-cod-edit-view.js?ver=2.5.1';
import './pk-quick-order-insurance-edit-view.js?ver=2.5.1';
import './pk-quick-order-description-edit-view.js?ver=2.5.1';
import './pk-quick-order-services-view.js?ver=2.5.1';

export class PkQuickOrderView extends LitElement {

    static accountInfoCache;

    static EVALUATE_DEBOUNCE_TIME = 1000;
    static EVALUATE_TRIGGER_FIELDS = [
        'shipmenttype',
        'packs',
        'cod',
        'insurance',
        'sender',
        'recipient',
        'courierservice',
    ];

    static styles = [
        globalStyles,
        styles,
    ];

    static properties = {
        orderId: {type: Number},
        isOpened: {type: Boolean, state: true},
        isOrderRequested: {type: Boolean, state: true},
        isSelectCourierOpened: {type: Boolean, state: true},
        order: {state: true},
        model: {state: true},
        pickupDates: {state: true},
        isPickupDatesRequested: {type: Boolean, state: true},
        isOrderEvaluationRequested: {type: Boolean, state: true},
        orderEvaluation: {state: true},
        isSaveRequested: {type: Boolean, state: true},
        createdOrder: {state: true},
        isAccountInfoRequested: {state: true},
        accountInfo: {state: true},
        editAddress: {state: true},
        isEditCodEnabled: {state: true},
        isEditInsuranceEnabled: {state: true},
        isDescriptionEditEnabled: {type: Boolean, state: true},
    }

    orderEvaluationRequest;

    addressesProvider = ProviderController.create(this, new SenderAddressesProvider(), {
        events: {
            load: () => {
                this.ensureSenderAddress();
                this.ensureCoverAddress();
            }
        }
    });

    templatesProvider = ProviderController.create(this, new TemplatesProvider(), {
        events: {
            load: this.ensureDefaultTemplate.bind(this),
        }
    });

    couriersProvider = ProviderController.create(this, new CouriersProvider(), {
        events: {
            load: this.onChangeCourier.bind(this),
        }
    });

    courierMessagesProvider = ProviderController.create(this, new CourierMessagesProvider());

    bankAccountsProvider = ProviderController.create(this, new BankAccountsProvider(), {load: false});

    constructor() {
        super();
        this.model = {
            form_version: 2,
        };
        this.accountInfo = this.constructor.accountInfoCache;
        this.setModelDefaultsFromAccountInfo();

        this.evaluateOrderDebounced = debounce(() => this.fetchOrderEvaluation(), this.constructor.EVALUATE_DEBOUNCE_TIME);
    }

    get courier() {
        return this.couriersProvider.get(this.model.courier);
    }

    get courierMessages() {
        return this.courierMessagesProvider.getForCourier(this.model.courier);
    }

    get hasNoCourierOrder() {
        return this.courier?.additional_data?.pickup?.nocourierorder?.available === true ||
            this.model?.courierservice?.LABELLESS === true;
    }

    get hasPickupDate() {
        return this.courier?.additional_data?.pickup?.pickupdate?.required === true &&
            this.model?.courierservice?.LABELLESS !== true;
    }

    get hasPickupTime() {
        return this.model?.courierservice?.LABELLESS !== true && (
            this.courier?.additional_data?.pickup?.pickuptimefrom?.required === true ||
            this.courier?.additional_data?.pickup?.pickuptimeto?.required === true
        );
    }

    resetDefaultCourier() {
        this.setModel({
            courier: this.order.courier,
        });
    }

    setModel(props) {
        this.model = {
            ...this.model,
            ...props,
        };
    }

    setModelDeepProperty(propertyName, props) {
        this.model = {
            ...this.model,
            [propertyName]: {
                ...(this.model[propertyName] || {}),
                ...props,
            },
        };
    }

    setTemplate(template) {
        const packs = [{
            length: template.length || 0,
            width: template.width || 0,
            height: template.height || 0,
            weight: template.weight || 0,
            amount: Math.max(1, template.amount),
            type: template.type,
        }];

        this.setModel({
            templateId: template.id,
            shipmenttype: template.shipment_type,
            courier: template.courier || this.model.courier,
            packs,
        });
    }

    async fetchAccountInfo() {
        this.isAccountInfoRequested = true;
        try {
            this.accountInfo = await ajaxRequest('/polkurier/account_info');
            this.constructor.accountInfoCache = this.accountInfo;
            this.setModelDefaultsFromAccountInfo();
            this.bankAccountsProvider.preload(this.accountInfo?.bank_accounts || []);
        } catch (e) {
            showErrorPopup(e);
        }
        this.isAccountInfoRequested = false;
    }

    setModelDefaultsFromAccountInfo() {
        if (!this.accountInfo) {
            return;
        }

        console.warn('setModelDefaultsFromAccountInfo', this.accountInfo);

        // Ustawia domyślne opcje
        this.setModel({
            courier: this.model?.courier || this.order?.courier || this.accountInfo.default_courier,
            cod: {
                codtype: this.accountInfo.default_cod_type,
                codbankaccount: this.accountInfo.default_cod_return_bank_account,
                return_cod: this.accountInfo.default_cod_return_type,
                ...(this.model?.cod || {}),
            },
            courierservice: {
                ...this.accountInfo.default_services,
                ...(this.model?.courierservice || {}),
            },
        });

        if (this.accountInfo.default_delivery_pickup_myself) {
            this.setModel({
                pickup: {
                    nocourierorder: true,
                },
            });
        }

    }

    async fetchOrder() {
        this.isOrderRequested = true;
        try {
            this.order = await ajaxRequest('/polkurier/shop_orders', {id: this.orderId});
            this.setModel({
                order_id: this.order.id,
                total_value: this.order.total_value,
                courier: this.order.courier,
                shipmenttype: this.order.shipmenttype,
                description: this.order.description,
                cod: {
                    ...(this.model?.cod || {}),
                    codamount: this.order.cod,
                },
                insurance: this.order.insurance,
                recipient: this.order.recipient,
            });
        } catch (e) {
            showErrorPopup(e);
        }
        this.isOrderRequested = false;
    }

    connectedCallback() {
        super.connectedCallback();
        this.isOpened = true;
    }

    updated(t) {
        super.updated(t);
        const didModelChange = t.has('model');
        const previousModel = t.get('model');

        if (didModelChange && previousModel?.courier !== this.model?.courier) {
            this.onChangeCourier();
        }

        if (didModelChange && previousModel?.courierservice?.LABELLESS !== this.model?.courierservice?.LABELLESS) {
            this.ensurePickupDate();
        }

        if (didModelChange && previousModel?.pickup?.nocourierorder !== this.model?.pickup?.nocourierorder) {
            this.ensurePickupDate();
            this.ensureSenderPickupPoint();
        }

        if (didModelChange && !previousModel?.pickup?.nocourierorder && this.model?.sender?.point_id) {
            this.setModelDeepProperty('sender', {point_id: null});
        }

        if (didModelChange && previousModel?.courierservice?.COVER_ADDRESS_SENDER !== this.model?.courierservice?.COVER_ADDRESS_SENDER) {
            this.ensureCoverAddress();
        }

        if (didModelChange && this.canEvaluateOrder() && this.shouldEvaluateOrder(previousModel)) {
            this.evaluateOrderDebounced();
        }
    }

    canEvaluateOrder() {
        return this.model &&
            this.model.shipmenttype &&
            this.model.packs &&
            this.model.packs.length > 0 &&
            this.model.recipient?.postcode &&
            this.model.sender?.postcode;
    }

    shouldEvaluateOrder(previousModel) {
        for (let field of this.constructor.EVALUATE_TRIGGER_FIELDS) {
            if (previousModel[field] !== this.model[field]) {
                return true;
            }
        }
        return false;
    }

    async fetchOrderEvaluation() {
        if (!this.canEvaluateOrder()) {
            return;
        }

        if (this.orderEvaluationRequest) {
            if (!this.orderEvaluationScheduledTimeout) {
                this.orderEvaluationScheduledTimeout = setTimeout(() => {
                    this.fetchOrderEvaluation();
                    this.orderEvaluationScheduledTimeout = undefined;
                }, this.constructor.EVALUATE_DEBOUNCE_TIME);
            }
            return;
        }

        this.isOrderEvaluationRequested = true;
        try {
            this.orderEvaluationRequest = ajaxRequest('/polkurier/orders/evaluate', this.model);
            this.orderEvaluation = await this.orderEvaluationRequest;
        } catch (e) {
            showErrorPopup(e);
        } finally {
            this.orderEvaluationRequest = undefined;
        }
        this.isOrderEvaluationRequested = false;
    }

    getCourierEvaluation(courier) {
        if (courier && this.orderEvaluation) {
            for (let evaluation of this.orderEvaluation) {
                if (evaluation.servicecode === courier) {
                    return evaluation;
                }
            }
        }
    }

    async fetchCourierPickupDates() {
        this.isPickupDatesRequested = true;
        try {
            this.pickupDates = await ajaxRequest('/polkurier/couriers/pickup_date', {
                courier: this.model?.courier,
                shipfrom: this.model?.sender?.postcode,
                shipto: this.model?.recipient?.postcode,
                parcel: this.model?.shipmenttype,
            });
        } catch (e) {
        }
        this.isPickupDatesRequested = false;
    }

    onChangeCourier() {
        const sender = {};
        const pickup = {
            pickupdate: null,
            pickuptimefrom: null,
            pickuptimeto: null,
        };

        if (this.courier) {
            if (!this.hasNoCourierOrder && this.model.pickup?.nocourierorder) {
                pickup.nocourierorder = false;
                sender.point_id = null;
            }

            if (!this.model.pickup?.nocourierorder && this.hasNoCourierOrder) {
                pickup.nocourierorder = this.accountInfo?.default_delivery_pickup_myself || false;
            }

            if (!this.hasPickupDate) {
                pickup.nocourierorder = true;
                pickup.multiPickup = false;
            }
        }

        this.pickupDates = undefined;
        this.setModelDeepProperty('pickup', pickup);
        this.setModelDeepProperty('sender', sender);
        this.ensurePickupDate();
        this.ensureSenderPickupPoint();
    }

    ensureSenderAddress() {
        this.setModelDeepProperty('sender', this.addressesProvider.getDefaultSenderAddress());
    }

    ensureCoverAddress() {
        if (this.model?.courierservice?.COVER_ADDRESS_SENDER === true) {
            if (!this.model?.cover_address) {
                this.setModel({cover_address: this.addressesProvider.getDefaultCoverAddress()});
            }
        } else {
            this.setModel({cover_address: null});
        }
    }

    ensureDefaultTemplate() {
        if (!this.model.packs) {
            const template = this.templatesProvider.find(m => m.default === true);
            if (template) {
                this.setTemplate({
                    ...template,
                    courier: this.model.courier ? this.model.courier : template.courier,
                });
            }
        }
    }

    ensurePickupDate() {
        if (this.model?.courierservice?.LABELLESS) {
            this.setModelDeepProperty('pickup', {
                nocourierorder: true,
                multiPickup: false,
            });
            return;
        }

        if (this.hasPickupDate) {
            if (this.pickupDates !== undefined) {
                if (!this.model.pickup?.nocourierorder && !this.model.pickup?.pickupdate && this.pickupDates?.length) {
                    const pickupDate = this.pickupDates[0];
                    this.setModelDeepProperty('pickup', {
                        pickupdate: pickupDate.pickupdate,
                        pickuptimefrom: pickupDate.time?.[0]?.timefrom,
                        pickuptimeto: pickupDate.time?.[0]?.timeto,
                    });
                }
            } else {
                if (this.model?.courier && this.model?.sender?.postcode) {
                    this.fetchCourierPickupDates().then(() => {
                        this.ensurePickupDate();
                    });
                }
            }
        }
    }

    ensureSenderPickupPoint() {
        if (!this.model?.sender) {
            return;
        }

        if (!this.model?.pickup?.nocourierorder || !isSenderPointRequired(this.model?.courier)) {
            this.setModelDeepProperty('sender', {
                point_id: null,
                delivery_point: null,
            });
            return;
        }

        const provider = getBaseCourier(this.model.courier);
        if (this.model?.sender?.delivery_point?.provider !== provider) {
            const defaultPoint = this.accountInfo?.default_pickup_point?.[provider];
            if (defaultPoint?.code) {
                this.setModelDeepProperty('sender', {
                    point_id: defaultPoint.code,
                    delivery_point: {
                        ...defaultPoint,
                        provider,
                    },
                });
            } else {
                this.setModelDeepProperty('sender', {
                    point_id: null,
                    delivery_point: null,
                });
            }
        }
    }

    firstUpdated() {
        super.firstUpdated();

        if (!this.accountInfo) {
            this.fetchAccountInfo().then(() => {
                this.ensureSenderPickupPoint();
            });
        }

        this.fetchOrder().then(() => {
            this.ensureSenderAddress();
            this.ensureCoverAddress();
            this.ensureDefaultTemplate();
            this.ensurePickupDate();
        });
    }

    async createOrder() {
        this.isSaveRequested = true;
        try {
            this.createdOrder = await ajaxRequest('/polkurier/orders/crete', this.model);
            this.dispatchEvent(new CustomEvent('createorder', {
                detail: this.createdOrder,
                bubbles: true,
                composed: true,
            }));
        } catch (e) {
            showErrorPopup(e);
        }
        this.isSaveRequested = false;
    }

    get isViewReady() {
        return this.isOrderRequested && this.addressesProvider.isFetched();
    }

    render() {
        const isDataRequested = this.isViewReady;
        const courierMessages = this.courierMessages;
        return html`
            <pk-dialog
                size="${MODAL_SIZE_EXTRA_LARGE}"
                .opened="${this.isOpened}"
            >
                <pk-grid style="--gap: 0.25rem">
                    <pk-col class="order-column">
                        <div class="pk-heading">
                            Przesyłka:
                        </div>
                        <pk-card>
                            <pk-loading-view .loading="${isDataRequested}">
                                <pk-quick-order-pack-view
                                    .order="${this.model}"
                                    .shopOrder="${this.order}"
                                    .evaluation="${this.getCourierEvaluation(this.model?.courier)}"
                                    .isEvaluated="${!!this.orderEvaluation}"
                                    .isEvaluationRequested="${this.isOrderEvaluationRequested}"
                                    .showResetDefaultCourierButton="${this.model.courier && this.order?.courier && this.order.courier !== this.model.courier}"
                                    @change="${(e) => this.setModel(e.target.order)}"
                                    @clickResetDefaultCourier="${this.resetDefaultCourier}"
                                    @clickEditButton="${() => this.isSelectCourierOpened = true}"
                                    @clickEditCoverAddressButton="${() => this.editAddress = 'COVER_ADDRESS'}"
                                    @clickEditCodButton="${() => this.isEditCodEnabled = true}"
                                    @clickEditInsuranceButton="${() => this.isEditInsuranceEnabled = true}"
                                    @clickEditDescriptionButton2="${() => this.isDescriptionEditEnabled = true}"
                                    @clickEditDescriptionButton="${() => this.isDescriptionEditEnabled = true}"
                                >
                                </pk-quick-order-pack-view>
                            </pk-loading-view>
                        </pk-card>
                        <div class="pk-heading">
                            Usługi dodatkowe:
                        </div>
                        <pk-card>
                            <pk-loading-view .loading="${isDataRequested}">
                                <pk-quick-order-services-view
                                    .order="${this.model}"
                                    @change="${(e) => this.setModel(e.target.order)}"
                                    @clickEditCoverAddressButton="${() => this.editAddress = 'COVER_ADDRESS'}"
                                >
                                </pk-quick-order-services-view>
                            </pk-loading-view>
                        </pk-card>
                        <div class="pk-heading">
                            Nadawca:
                            ${this.addressesProvider.length > 0 ? html`
                                <button class="btn btn-raised btn-rounded btn-icon" @click="${() => this.editAddress = 'SENDER'}" title="Zmień adres nadawcy">
                                    <pk-icon icon="pencil" size="text" type="plain" ></pk-icon>
                                </button>
                            ` : null}
                        </div>
                        <pk-card>
                            <pk-loading-view .loading="${isDataRequested}">
                                ${this.addressesProvider.isFetched() && this.addressesProvider.length === 0 ? html`
                                    <div class="d-flex gap align-items-center text-danger">
                                        <pk-icon icon="parcel-problem" class="text-danger"></pk-icon>
                                        <p>
                                            Nie masz zdefiniowanych adresów nadania.<br>
                                            Przejdź do <a href="admin.php?page=polkurier&tab=addresses" target="_blank">ustawień</a>, aby dodać nowy adres.
                                        </p>
                                    </div>
                                ` : html`
                                    <pk-address type="sender" .model="${this.model.sender}">
                                        <pk-col style="--size-lg: 6;" slot="point">
                                            <pk-order-pickup-view
                                                .courier="${this.model?.courier}"
                                                .pickup="${this.model.pickup}"
                                                .sender="${this.model?.sender}"
                                                .pickupDates="${this.pickupDates}"
                                                .pickupPoint="${this.model?.sender?.delivery_point}"
                                                .hasNoCourierOrder="${this.hasNoCourierOrder}"
                                                .hasPickupDate="${this.hasPickupDate}"
                                                .hasPickupTime="${this.hasPickupTime}"
                                                @change="${e => {
                                                    this.setModelDeepProperty('pickup', e.target.pickup);
                                                    this.setModelDeepProperty('sender', e.target.sender);
                                                }}"
                                            ></pk-order-pickup-view>
                                        </pk-col>
                                    </pk-address>
                                `}
                            </pk-loading-view>
                        </pk-card>
                        <div class="pk-heading">
                            Odbiorca:
                            <button class="btn btn-raised btn-rounded btn-icon" @click="${() => this.editAddress = 'RECIPIENT'}" title="Zmień adres nadawcy">
                                <pk-icon icon="pencil" size="text" type="plain" ></pk-icon>
                            </button>
                        </div>
                        <pk-card>
                            <pk-loading-view .loading="${isDataRequested}">
                                <pk-address type="recipient" .model="${this.model.recipient}"></pk-address>
                            </pk-loading-view>
                        </pk-card>
                        
                        ${courierMessages?.length > 0 ? html`
                            <div class="d-flex-column mt gap-sm">
                                ${courierMessages.map(m => html`
                                    <pk-alert .type="${m.severity}">${unsafeHTML(m.message)}</pk-alert>
                                `)}
                            </div>
                        ` : null}
                    </pk-col>
                    <pk-col class="templates-column">
                        ${this.createdOrder ? html`
                            <pk-quick-created-order-view
                                .createdOrder="${this.createdOrder}"
                            ></pk-quick-created-order-view>
                        ` : html`
                            <div class="pk-heading">
                                Wybierz szablon:
                            </div>
                            <div style="flex-shrink: 1; overflow: auto; padding-bottom: 0.5rem;">
                                <pk-quick-order-templates
                                    .value="${this.model.templateId}"
                                    @change="${e => this.setTemplate(e.target.template)}"
                                ></pk-quick-order-templates>
                            </div>
                            <div class="d-flex-column gap" style="flex-shrink: 0; padding: 1.5rem 0 0.5rem 0; margin-top: auto; text-align: center;">
                                ${this.model?.cod?.codamount > 0 &&  this.model?.cod?.return_cod === 'BA' && !this.model?.cod?.codbankaccount ? html`
                                    <pk-alert type="danger">
                                        Brak numeru konta do zwrotu pobrania.
                                        <button class="btn btn-link" @click="${() => this.isEditCodEnabled = true}">
                                            Edytuj parametry pobrania i ustaw konto
                                        </button>
                                    </pk-alert>
                                ` : null}
                                <button 
                                    class="btn btn-lg btn-primary btn-block"
                                    ?disabled="${isDataRequested || this.isSaveRequested || this.isPickupDatesRequested}"
                                    @click="${this.createOrder}"
                                    style="width: 100%"
                                >
                                    ${this.isSaveRequested ? html`
                                        <pk-spinner size="sm"></pk-spinner>
                                    ` : null}
                                    Zamów przesyłkę
                                </button>
                            </div>
                        `}
                    </pk-col>
                </pk-grid>
            </pk-dialog>
            
            ${this.isSelectCourierOpened ? html`
                <pk-dialog
                    .opened="${this.isSelectCourierOpened}"
                    @closed="${(e) => {
                        e.stopPropagation();
                        this.isSelectCourierOpened = false
                    }}"
                >
                    <pk-quick-courier-view
                        .order="${this.model}"
                        .orderEvaluation="${this.orderEvaluation}"
                        .isEvaluationRequested="${this.isOrderEvaluationRequested}"
                        @select="${e => {
                            this.setModel({courier: e.detail.servicecode});
                            closeParentDialog(e.target);
                        }}"
                    ></pk-quick-courier-view>
                </pk-dialog>
            ` : null}
            
            ${this.editAddress ? html`
                <pk-dialog
                    .opened="${!!this.editAddress}"
                    size="${this.editAddress === 'RECIPIENT' ? MODAL_SIZE_DEFAULT : MODAL_SIZE_MEDIUM}"
                    @closed="${(e) => {
                        e.stopPropagation();
                        this.editAddress = undefined
                    }}"
                >
                    ${this.editAddress === 'SENDER' ? html`
                        <pk-quick-order-address-select-view
                            .address="${this.model?.sender}"
                            .addresses="${this.addressesProvider.getSenderAddresses()}"
                            @select="${(e) => {
                                this.setModel({sender: e.detail});
                                closeParentDialog(e.target);
                            }}"
                        ></pk-quick-order-address-select-view>
                    ` : null}
                    
                    ${this.editAddress === 'COVER_ADDRESS' ? html`
                        <pk-quick-order-address-select-view
                            .address="${this.model?.cover_address}"
                            .addresses="${this.addressesProvider.getCoverAddresses()}"
                            @select="${(e) => {
                                this.setModel({cover_address: e.detail});
                                closeParentDialog(e.target);
                            }}"
                        ></pk-quick-order-address-select-view>
                    ` : null}
    
                    ${this.editAddress === 'RECIPIENT' ? html`
                        <pk-quick-order-recipient-edit-view
                            .address="${this.model?.recipient}"
                            @confirm="${(e) => {
                                this.setModelDeepProperty('recipient', e.target.address);
                                closeParentDialog(e.target);
                            }}"
                            @cancel="${(e) => closeParentDialog(e.target)}"
                        ></pk-quick-order-recipient-edit-view>
                    ` : null}
                </pk-dialog>
            ` : null}

            ${this.isEditCodEnabled ? html`
                <pk-dialog
                    size="${MODAL_SIZE_SMALL}"
                    .opened="${this.isEditCodEnabled}"
                    @closed="${(e) => {
                        e.stopPropagation();
                        this.isEditCodEnabled = false
                    }}"
                >
                    <pk-quick-order-cod-edit-view
                        .cod="${{
                            codtype: this.accountInfo?.default_cod_type,
                            return_cod: this.accountInfo?.default_cod_return_type,
                            codbankaccount: this.accountInfo?.default_cod_return_bank_account,
                            ...(this.model?.cod || {}),
                        }}"
                        @confirm="${this.#onConfirmChangeCod}"
                        @cancel="${(e) => closeParentDialog(e.target)}"
                    ></pk-quick-order-cod-edit-view>
                </pk-dialog>
            ` : null}
            
            ${this.isEditInsuranceEnabled ? html`
                <pk-dialog
                    size="${MODAL_SIZE_SMALL}"
                    .opened="${this.isEditInsuranceEnabled}"
                    @closed="${(e) => {
                        e.stopPropagation();
                        this.isEditInsuranceEnabled = false
                    }}"
                >
                    <pk-quick-order-insurance-edit-view
                        .value="${this.model?.insurance}"
                        @confirm="${async (e) => {
                            this.setModel({insurance: e.target.value});
                            closeParentDialog(e.target);
                        }}"
                        @cancel="${(e) => closeParentDialog(e.target)}"
                    ></pk-quick-order-insurance-edit-view>

                </pk-dialog>
            ` : null}
            
            ${this.isDescriptionEditEnabled ? html`
                <pk-dialog
                    size="${MODAL_SIZE_SMALL}"
                    .opened="${this.isDescriptionEditEnabled}"
                    @closed="${(e) => {
                        e.stopPropagation();
                        this.isDescriptionEditEnabled = false
                    }}"
                >
                    <pk-quick-order-description-edit-view
                        .value="${this.model?.description}"
                        @confirm="${async (e) => {
                            this.setModel({description: e.target.value});
                            closeParentDialog(e.target);
                        }}"
                        @cancel="${(e) => closeParentDialog(e.target)}"
                    ></pk-quick-order-description-edit-view>
                </pk-dialog>
            ` : null}
        `;
    }

    // Akcja po zatwierdzeniu zmiany COD
    async #onConfirmChangeCod(e) {
        const target = e.target;
        const cod =  target.cod;
        let insurance = numberFilter(this.model?.insurance) || 0;
        if (
            cod.codamount > 0 &&
            cod.codamount > insurance &&
            await createConfirm('', `
                Wartość pobrania <strong>${formatCurrency(cod.codamount)} zł</strong> jest większa niż warość ubezpieczenia <strong>${formatCurrency(insurance)} zł</strong>.<br><br> 
                Czy Zwiększyć wartość ubezpieczenia do wartości pobrania?
            `, {
                cancelLabel: 'Nie',
            })
        ) {
            insurance = cod.codamount;
            createToast(`Zmianiono wartość ubezpieczenia na <strong>${formatCurrency(insurance)} zł</strong>`, {
                type: TOAST_TYPE_SUCCESS,
            });
        }
        this.setModel({cod, insurance});
        closeParentDialog(target);
    }

}

customElements.define('pk-quick-order-view', PkQuickOrderView);
