import {LitElement, html, css, ref, createRef} from "../libs/lit.dist.js?ver=2.5.1";

let inputId = 1;

export class PkToggle extends LitElement {

    inputId;
    inputRef = createRef();

    static formAssociated = true;

    static styles = css`
        * {
            box-sizing: border-box;
        }
        
        :host {
            box-sizing: border-box;
            --color: var(--pk-gray-500);
            --accent: var(--pk-primary-500);
            --background: white;
            --toggle-color: var(--color);
            --border-width: 2px;
            --toggle-gap-y: 4px;
            --toggle-gap-x: 2px;
            --gap: 0.5rem;
            --width: 2rem;
            --height: 1.125rem;
            --toggle-size: calc(var(--height) - (var(--toggle-gap-y) * 2));
            display: inline-flex;
            gap: var(--gap);
            margin: 0;
            padding: 0;
            align-items: baseline;
        }
        
        input {
            flex-shrink: 0;
            top: var(--input-offset, 0.123em);
            margin: 0;
            padding: 0;
            vertical-align: top;
            appearance: none;
            background-color: var(--background);
            border-radius: 2em;
            width: var(--width);
            height: var(--height);
            border: var(--border-width) solid var(--color);
            position: relative;
            transition: all 0.15s ease-in-out;
        }
        
        input::after {
            content: '';
            background-color: var(--toggle-color);
            position: absolute;
            top: 50%;
            left: var(--toggle-gap-x);
            translate: 0 -50%;
            width: var(--toggle-size);
            height: var(--toggle-size);
            border-radius: 50%;
            display: block;
            transition: all 0.15s ease-in-out;
        }
        
        input:checked::after {
            left: calc(var(--width) - var(--toggle-size) - (2 * var(--toggle-gap-x)));
            --toggle-color: white;
        }
        
        input:checked {
            --background: var(--accent);
            --color: var(--accent);
        }
        
        input:focus {
            --toggle-color: var(--color);
            box-shadow: 0 0 0 .25rem color(from var(--pk-input-active-color) xyz calc(x * 1.3) calc(y * 1.3) calc(z * 1.3) / 0.3)
        }
         
        :host(.pk-toggle-small) {
            --width: 1.5rem;
            --height: 0.9rem;
            --toggle-gap-y: 3px;
            --input-offset: 0.256em;
        }
             
        :host(.pk-toggle-align-center) {
            align-items: center;
            top: 0;
        }
    
        :host(.pk-toggle-align-center) input {
            top: 0;
        }
    
        :host(.align-end) {
            flex-direction: row-reverse;
        }
        
        :host(.align-end.text-start) .label,
        :host(.align-end.text-left) .label {
            margin-right: auto;
        }
    `;

    static properties = {
        type: {type: String},
        name: {type: String},
        checked: {type: Boolean, reflect: true},
        disabled: {type: Boolean},
        value: {type: String},
        url: {type: String},
        isToggleRequested: {type: Boolean, state: true},
    };

    constructor() {
        super();
        this.inputId = inputId++;
        if (typeof this.attachInternals === 'function') {
            this.internals = this.attachInternals();
        }
        this.value = 'On';
    }

    connectedCallback() {
        super.connectedCallback();
        this.updateFromValue();
    }

    async onChangeHandler(e) {
        e.preventDefault();
        e.stopPropagation();
        if (!this.disabled) {
            if (this.dispatchEvent(new Event('beforechange', {bubbles: true, cancelable: true})) === false) {
                this.inputRef.value.checked = this.checked;
                return;
            }
            this.checked = this.inputRef.value.checked;
            this.dispatchEvent(new Event('input', {bubbles: true}));
            this.dispatchEvent(new Event('change', {bubbles: true}));
        }
    }

    updateFromValue() {
        if (this.internals) {
            this.internals.setFormValue(this.checked && !this.disabled ? this.value : undefined);
        }
    }

    update(t) {
        super.update(t);
        if (t.has('checked') || t.has('disabled')) {
            this.updateFromValue();
        }
    }

    render() {
        return html`
            <input
                ${ref(this.inputRef)}
                type="checkbox"
                role="switch"
                id="pk-live-checkbox-internal-${this.inputId}"
                .checked="${this.checked}"
                .disabled="${this.disabled || this.isToggleRequested}"
                @change="${e => this.onChangeHandler(e)}"
            >
            <div class="label">
                <label for="pk-live-checkbox-internal-${this.inputId}">
                    <slot></slot>
                </label>
                <div>
                    <slot name="description"></slot>
                </div>
            </div>
        `;
    }
}

customElements.define('pk-toggle', PkToggle);
