<?php

namespace Polkurier;

use Exception;
use Polkurier\Exception\ApiException;
use Polkurier\Normalizer\Denormalizer;
use ReflectionClass;
use ReflectionException;
use stdClass;

/**
 * @template T
 */
abstract class AbstractApiObjectManager
{

    private static array $reflectionCache = [];
    private static array $dataCache = [];

    protected const MODEL_CLASS = stdClass::class;
    protected const TRANSIENT_KEY = '';
    protected const TRANSIENT_TTL = 0;

    /**
     * @param string $id
     * @return T|null
     */
    public function get(string $id): ?object
    {
        $this->load();
        $data = $this->getDataCache('dataByPk');
        return $data[$id] ?? null;
    }

    /**
     * @return T[]
     */
    public function getAll(): array
    {
        $this->load();
        return $this->getDataCache('data') ?? [];
    }

    /**
     * @return bool
     */
    protected function load(): bool
    {
        $cachedData = $this->getDataCache('data');
        if ($cachedData !== null) {
            return false;
        }

        try {
            $data = get_transient(static::TRANSIENT_KEY);
            if ($data === false) {
                $data = $this->fetchApiData();
                if (static::TRANSIENT_TTL > 0) {
                    set_transient(static::TRANSIENT_KEY, $data, static::TRANSIENT_TTL);
                }
            }

            $items = [];
            $itemsByPk = [];

            $normalizer = new Denormalizer();
            foreach ($data as $row) {
                $object = $this->createModelInstance();
                $normalizer->denormalize($object, $row);
                $items[] = $object;

                $primaryKeyValue = $this->getModelPrimaryKeyValue($object);
                if ($primaryKeyValue !== null) {
                    $itemsByPk[$this->getModelPrimaryKeyValue($object)] = $object;
                }
            }

            $this->setData('data', $items);
            $this->setData('dataByPk', $itemsByPk);
            return true;
        } catch (Exception $e) {
            $logger = wc_get_logger();
            if ($logger !== null) {
                $logger->critical($e->getMessage(), $e->getTrace());
            }
        }
        return false;
    }

    protected function getDataCache(string $key): ?array
    {
        return self::$dataCache[static::TRANSIENT_KEY][$key] ?? null;
    }

    protected function setData(string $key, array $data): void
    {
        self::$dataCache[static::TRANSIENT_KEY][$key] = $data;
    }

    /**
     * @return T
     * @throws ReflectionException
     */
    protected function createModelInstance(): object
    {
        if (!isset(static::$reflectionCache[static::MODEL_CLASS])) {
            self::$reflectionCache[static::MODEL_CLASS] = new ReflectionClass(static::MODEL_CLASS);
        }
        return self::$reflectionCache[static::MODEL_CLASS]->newInstanceWithoutConstructor();
    }

    /**
     * @param T $object
     * @return string|int|null
     */
    abstract protected function getModelPrimaryKeyValue($object);

    /**
     * @return array
     * @throws ApiException
     */
    abstract protected function fetchApiData(): array;

}
