<?php

namespace Polkurier\Controller;


use Polkurier\Polkurier;
use Polkurier\Exception\HttpException;
use Polkurier\Http\JsonResponse;
use Polkurier\Http\Request;
use Polkurier\Order\OrderShippingPointManager;
use Polkurier\Types\ShipmentType;
use Polkurier\Util\OrderTools;

class ShopOrderController
{

    /**
     * @throws HttpException
     */
    public function indexAction(Request $request): JsonResponse
    {
        $orderId = $request->request->getString('id');
        $order = wc_get_order($orderId);
        if (empty($order)) {
            return new JsonResponse([]);
        }

        $order_data = $order->get_data();
        $shippingPoint = (new OrderShippingPointManager())->get($orderId);

        $selectedCarrier = '';
        if ($shippingPoint !== null) {
            $selectedCarrier = $shippingPoint->provider;
        }

        if (empty($selectedCarrier)) {
            $shippingMethod = OrderTools::getShippingMethod($order);
            if ($shippingMethod !== null && OrderTools::isOwnShippingMethod($shippingMethod)) {
                $selectedCarrier = $shippingMethod->getPolkurierId();
            }
        }

        if (empty($selectedCarrier)) {
            $selectedCarrier = (string)get_option(Polkurier::NAME . '_default_carrier_id');
        }

        // Jeśli InPost zagraniczny
        if ($selectedCarrier === 'INPOST_PACZKOMAT' && $order_data['shipping']['country'] !== 'PL') {
            $selectedCarrier = 'INPOST_INTERNATIONAL';
        }

        $codAmount = 0;
        if ($order->get_payment_method() === 'cod') {
            $codAmount = $order->get_total();
        }

        $insurance = 0;
        if ((int)get_option('polkurier_default_courierservice_INSURANCE') === 1) {
            $insurance = $order->get_total();
        }

        return new JsonResponse([
            'id' => $orderId,
            'shipmenttype' => ShipmentType::BOX,
            'courier' => $selectedCarrier,
            'description' => "Zamówienie #$orderId",
            'cod' => $codAmount,
            'insurance' => $insurance,
            'total_value' => $order->get_total(),
            'recipient' => [
                'company' => $order_data['shipping']['company'],
                'person' => trim($order_data['shipping']['first_name'] . ' ' . $order_data['shipping']['last_name']),
                'street' => trim($order_data['shipping']['address_1'] . ' ' . $order_data['shipping']['address_2']),
                'city' => $order_data['shipping']['city'],
                'postcode' => $order_data['shipping']['postcode'],
                'country' => $order_data['shipping']['country'],
                'email' => $order_data['billing']['email'],
                'phone' => !empty($order_data['shipping']['phone']) ? $order_data['shipping']['phone'] : $order_data['billing']['phone'],
                'point_id' => $shippingPoint !== null ? $shippingPoint->code : null,
                'delivery_point' => $shippingPoint !== null ? [
                    'code' => $shippingPoint->code,
                    'label' => $shippingPoint->label,
                    'provider' => $shippingPoint->provider,
                    'type' => $shippingPoint->type,
                ] : null,
            ]
        ]);
    }

}
