<?php


namespace Polkurier\Http;

use UnexpectedValueException;

class Response
{

    protected $content;
    protected int $statusCode;
    protected array $headers;

    /**
     * Response constructor.
     * @param mixed $content
     * @param int $code
     * @param array $headers
     */
    public function __construct($content = '', int $code = 200, array $headers = [])
    {
        $this->setContent($content);
        $this->statusCode = $code;
        $this->headers = $headers;
    }

    public function setContent($content): self
    {
        if (null !== $content && !is_string($content) && !is_numeric($content) && !is_callable([$content, '__toString'])) {
            throw new UnexpectedValueException(sprintf('The Response content must be a string or object implementing __toString(), "%s" given.', gettype($content)));
        }
        $this->content = (string)$content;
        return $this;
    }

    /**
     * Sends HTTP headers.
     */
    public function sendHeaders(): self
    {
        if (headers_sent()) {
            return $this;
        }
        foreach ($this->headers as $name => $value) {
            header($name . ': ' . $value, true, $this->statusCode);
        }
        http_response_code($this->statusCode);
        return $this;
    }

    /**
     * Sends content for the current web response.
     */
    public function sendContent(): self
    {
        echo $this->content;
        return $this;
    }

    /**
     * Sends HTTP headers and content.
     */
    public function send(): self
    {
        $this->sendHeaders();
        $this->sendContent();
        return $this;
    }

    public function setHeader(string $name, string $value): self
    {
        $this->headers[$name] = $value;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getContent()
    {
        return $this->content;
    }

    public function getStatusCode(): int
    {
        return $this->statusCode;
    }

    public function getHeaders(): array
    {
        return $this->headers;
    }

}
