<?php

namespace Polkurier\Order;

use DateTimeImmutable;
use Polkurier\ApiClient;
use Polkurier\ConfigManager;
use Polkurier\Exception\ApiException;
use Polkurier\Types\ConfigType;
use Throwable;

class OrderCreator
{

    private int $form_version = 0;

    public function setFormVersion(int $form_version): OrderCreator
    {
        $this->form_version = $form_version;
        return $this;
    }

    /**
     * @param Order $order
     * @return Order
     * @throws ApiException
     * @throws Throwable
     */
    public function create(Order $order): Order
    {
        $config = new ConfigManager();
        $orderData = [
            'platform' => 2,
            'form_version' => $this->form_version,
            'external_application_id' => $config->getString(ConfigType::INSTALLATION_ID),
            'external_order_id' => $order->getShopOrderId(),
            'courier' => $order->getCourier(),
            'shipmenttype' => $order->getShipmentType(),
            'description' => $order->getDescription(),
            'sender' => $this->normalizeAddress($order->getSender()),
            'recipient' => $this->normalizeAddress($order->getRecipient()),
            'cover_address' => $this->normalizeAddress($order->getCoverAddress()),
            'packs' => array_map(static function (OrderPack $pack) {
                return [
                    'length' => $pack->length,
                    'width' => $pack->width,
                    'height' => $pack->height,
                    'weight' => $pack->weight,
                    'amount' => $pack->amount,
                    'type' => $pack->type,
                ];
            }, $order->getPacks()),
            'pickup' => $order->getPickup() !== null ? [
                'pickupdate' => $order->getPickup()->date !== null ? $order->getPickup()->date->format('Y-m-d') : null,
                'pickuptimefrom' => $order->getPickup()->timeFrom,
                'pickuptimeto' => $order->getPickup()->timeTo,
                'nocourierorder' => $order->getPickup()->noCourierOrder,
            ] : null,
            'COD' => $order->getCod() && $order->getCod()->amount > 0.0 ? [
                'codtype' => $order->getCod()->returnTime,
                'codamount' => $order->getCod()->amount,
                'codbankaccount' => $order->getCod()->bankAccount,
                'return_cod' => $order->getCod()->returnType,
            ] : null,
            'insurance' => $order->getInsurance(),
            'courierservice' => $order->getCourierServices(),
        ];

        $response = (new ApiClient())->makeApiRequest('create_order', $orderData);

        $order->setOrderNumber((string)$response['order_number']);
        $order->setLabel((array)$response['label']);
        $order->setPriceGross((float)$response['price_gross']);
        $order->setPriceNet((float)$response['price_net']);
        $order->setDateCreated(new DateTimeImmutable());
        $order->setAuthorId(get_current_user_id() ?: null);
        (new OrderManager())->save($order);
        return $order;
    }

    /**
     * @param OrderAddress|null $address
     * @return array|null
     */
    private function normalizeAddress(?OrderAddress $address): ?array
    {
        if ($address === null) {
            return null;
        }
        return [
            'company' => $address->company,
            'person' => $address->person,
            'street' => $address->street,
            'housenumber' => $address->housenumber,
            'flatnumber' => $address->flatnumber,
            'postcode' => $address->postcode,
            'city' => $address->city,
            'email' => $address->email,
            'phone' => $address->phone,
            'country' => $address->country,
            'point_id' => $address->point_id,
            'point_name' => $address->point_name,
        ];
    }

}
