<?php

namespace Polkurier;

use Polkurier\Blocks\WooCommerceBlocksManager;
use Polkurier\Environment\AdminEnvironment;
use Polkurier\Environment\FrontEnvironment;
use Polkurier\Integrations\AbstractIntegration;
use Polkurier\Integrations\FlexibleShipping;
use Polkurier\ShippingMethods\AbstractShippingMethod;
use Polkurier\ShippingMethods\DhlPopShippingMethodMethod;
use Polkurier\ShippingMethods\DpdShippingMethodMethod;
use Polkurier\ShippingMethods\FedExDtrShippingMethodMethod;
use Polkurier\ShippingMethods\InpostParcelMachineShippingMethodMethod;
use Polkurier\ShippingMethods\PaczkaWRuchuShippingMethodMethod;
use Polkurier\ShippingMethods\PocztexPointShippingMethod;
use Polkurier\ShippingMethods\UpsAccessPointShippingMethodMethod;
use Polkurier\Upgrade\PluginUpgradeManager;
use Polkurier\Util\Notice;
use WC_Shipping_Method;
use WC_Shipping_Zones;

if (!defined('POLKURIER_PLUGIN_PATH')) {
    define('POLKURIER_PLUGIN_PATH', wp_normalize_path(dirname(plugin_dir_path(__FILE__), 2) . '/'));
}

final class Polkurier
{

    public const NAME = 'polkurier';
    public const DIR = POLKURIER_PLUGIN_PATH;
    public const URL = POLKURIER_PLUGIN_URL;
    public const VERSION = POLKURIER_PLUGIN_VERSION;
    public const SLUG = POLKURIER_PLUGIN_SLUG;
    private static  bool $initialized = false;
    private static ?Polkurier $runtimeInstance = null;

    /**
     * @var string[]
     */
    private array $shippingMethods = [
        'pk_inpost_paczkomat' => InpostParcelMachineShippingMethodMethod::class,
        'pk_paczka_w_ruchu' => PaczkaWRuchuShippingMethodMethod::class,
        'pk_dpd_point' => DpdShippingMethodMethod::class,
        'pk_fedex_dtr' => FedExDtrShippingMethodMethod::class,
        'pk_dhl_point' => DhlPopShippingMethodMethod::class,
        'pk_ups_access_point' => UpsAccessPointShippingMethodMethod::class,
        'pk_pocztex_point' => PocztexPointShippingMethod::class,
    ];

    /**
     * @var AbstractIntegration[]
     */
    private array $integrations = [];

    public static function instance(): Polkurier
    {
        if (self::$runtimeInstance === null) {
            self::$runtimeInstance = new self();
        }
        return self::$runtimeInstance;
    }

    /**
     * @return array
     */
    public function getShippingMethods(): array
    {
        $shippingMethods = [];
        foreach ($this->getAllShippingMethods() as $shippingMethod) {
            if ($shippingMethod instanceof AbstractShippingMethod) {
                $shippingMethods[] = $shippingMethod;
            }
        }
        return $shippingMethods;
    }

    /**
     * @param $id
     * @param null $instanceId
     * @return WC_Shipping_Method|null
     */
    public function getShippingMethodById($id, $instanceId = null): ?WC_Shipping_Method
    {
        foreach ($this->getAllShippingMethods() as $shippingMethod) {
            if ($shippingMethod->id === $id && ($instanceId === null || $instanceId === $shippingMethod->instance_id)) {
                return $shippingMethod;
            }
        }
        return null;
    }

    /**
     * @return WC_Shipping_Method[]
     */
    public function getAllShippingMethods(): array
    {
        $zones = (array)WC_Shipping_Zones::get_zones();
        $shippingMethods = [];
        foreach ($zones as $zone) {
            /** @var WC_Shipping_Method $method */
            foreach ((array)$zone['shipping_methods'] as $method) {
                $shippingMethods[] = $method;
            }
        }
        return $shippingMethods;
    }

    /**
     * @return Array<AbstractShippingMethod|AbstractIntegration>
     */
    public function getMapShippingMethods(): array
    {
        $methods = [];
        foreach ($this->getAllShippingMethods() as $shippingMethod) {
            if ($shippingMethod instanceof AbstractShippingMethod && $shippingMethod->isParcelPickupPoint()) {
                $methods[] = $shippingMethod;
            }
            foreach ($this->integrations as $integration) {
                if ($integration instanceof FlexibleShipping && $integration->shouldShowMapButton($shippingMethod)) {
                    $methods[] = $shippingMethod;
                }
            }
        }
        return $methods;
    }

    /**
     * @return int[]
     */
    public function getMapShippingMethodsIds(): array
    {
        $methodsIds = [];
        foreach ($this->getMapShippingMethods() as $shippingMethod) {
            $methodsIds[] = $shippingMethod->id . ':' . $shippingMethod->instance_id;
        }
        return $methodsIds;
    }

    public function getCurrentShippingMethod(): ?WC_Shipping_Method
    {
        $session = WC()->session;
        if ($session === null) {
            return null;
        }

        $chosenShippingMethods = $session->chosen_shipping_methods;
        if (!is_array($chosenShippingMethods) || count($chosenShippingMethods) === 0) {
            return null;
        }

        $currentShippingMethodId = $chosenShippingMethods[0];
        foreach ($this->getAllShippingMethods() as $method) {
            $methodId = $method->id . ':' . $method->instance_id;
            if ($methodId === $currentShippingMethodId) {
                return $method;
            }
        }
        return null;
    }

    /**
     * @return AbstractIntegration[]
     */
    public function getIntegrations(): array
    {
        return $this->integrations;
    }

    public function run(): void
    {
        if (self::$initialized) {
            return;
        }

        add_action('plugins_loaded', function (): void {
            load_plugin_textdomain('polkurier', false, self::DIR . '/languages/');

            if (!RequirementsValidator::isWooCommerceValid()) {
                add_action('admin_notices', function () {
                    Notice::error('<strong>Polkurier</strong> wymaga do działania wtyczki WooCommerce w wersji minimum 8.3');
                });
                return;
            }

            $activePlugins = apply_filters('active_plugins', get_option('active_plugins'));

            // Integracja z FlexibleShipping
            if (in_array('flexible-shipping/flexible-shipping.php', $activePlugins, true)) {
                $this->integrations[] = new FlexibleShipping();
            }

            // Inicjalizacja metod wysyłki
            add_filter('woocommerce_shipping_methods', function (array $methods): array {
                foreach ($this->shippingMethods as $methodId => $methodClass) {
                    $methods[$methodId] = $methodClass;
                }
                return $methods;
            });
        }, 100);

        (new FrontEnvironment($this))->initialize();
        (new AdminEnvironment())->initialize();
        (new PluginUpgradeManager())->initialize();
        (new WooCommerceBlocksManager())->initialize();

        self::$initialized = true;
    }

}
