<?php

namespace Polkurier\ShippingMethods;

use Polkurier\MapsTokenManager;
use Polkurier\Polkurier;
use Polkurier\Types\CheckoutPointButtonPosition;
use Polkurier\Util\Numbers;
use Polkurier\View;
use WC_Order;

abstract class AbstractShippingMethod extends \WC_Shipping_Method
{

    public const ERROR_MESSAGE_NOT_POINT_SELECTED = 'ERROR_MESSAGE_NOT_POINT_SELECTED';
    public const ERROR_MESSAGE_NO_PHONE_NUMBER = 'ERROR_MESSAGE_NO_PHONE_NUMBER';

    /**
     * @var string
     */
    private $freeShippingFrom;

    /**
     * @var mixed
     */
    private $type;

    /**
     * @var string
     */
    public $tax_status = 'none';

    protected $fee_cost = 0;

    public function __construct($instance_id = 0)
    {
        $this->id = $this->getId();
        parent::__construct($instance_id);
        $this->method_title = __($this->getDefaultTitle(), 'polkurier');

        $this->supports = [
            'shipping-zones',
            'instance-settings',
            'instance-settings-modal',
        ];

        $this->init_form_fields();
        $this->init_settings();
        $this->init();
        $this->initActions();
    }

    public function init(): void
    {
        $this->title = $this->get_option('title');
        $this->tax_status = $this->get_option('tax_status');
        $this->fee = $this->get_option('fee');
        $this->freeShippingFrom = $this->get_option('free_shipping_from', '');
        $this->type = $this->get_option('type', 'class');
    }

    abstract protected function getDefaultTitle(): string;
    abstract public function getId(): string;
    abstract public function getPolkurierId(): string;
    abstract public function getSelectButtonLabel(): string;
    abstract public function getSelectedPointLabel(): string;
    abstract protected function getErrorMessages(): array;
    public function isParcelPickupPoint(): bool
    {
        return false;
    }

    protected function getErrorMessage(string $errorType): string
    {
        $messages = $this->getErrorMessages();
        return $messages[$errorType] ?? $errorType;
    }

    protected function isSelected(): bool
    {
        return in_array($this->id . ':' . $this->instance_id, WC()->session->get('chosen_shipping_methods'), true);
    }

    public function initActions(): void
    {
        add_action('woocommerce_update_options_shipping_' . $this->id, [$this, 'process_admin_options']);
    }

    public function init_form_fields()
    {
        $namespace = 'polkurier';
        $cost_desc = __('Enter a cost (excl. tax) or sum, e.g. <code>10.00 * [qty]</code>.', 'woocommerce') . '<br/><br/>' . __('Use <code>[qty]</code> for the number of items, <br/><code>[cost]</code> for the total cost of items, and <code>[fee percent="10" min_fee="20" max_fee=""]</code> for percentage based fees.', 'woocommerce');
        $settings = array(
            'title' => [
                'title' => __('Nazwa wyświetlana', $namespace),
                'type' => 'text',
                'default' => __($this->getDefaultTitle(), $namespace),
                'desc_tip' => false
            ],
            'cost' => [
                'title' => __('Cena', $namespace),
                'type' => 'text',
                'custom_attributes' => [
                    'step' => 'any',
                    'min' => '0'
                ],
                'default' => '0',
                'description' => $cost_desc,
                'placeholder' => '0.00',
                'desc_tip' => true,
                'sanitize_callback' => array($this, 'sanitize_cost'),
            ],
            'cost_cod' => [
                'title' => __('Dopłata za pobranie', $namespace),
                'type' => 'text',
                'custom_attributes' => [
                    'step' => 'any',
                    'min' => '0'
                ],
                'default' => '',
                'placeholder' => '0.00',
                'desc_tip' => __('Dodatkowa opłata za pobranie doliczana do ceny podstawowej w przypadku wybrania płatności za pobraniem', $namespace),
            ],
            'tax_status' => array(
                'title'   => __( 'Tax status', 'woocommerce' ),
                'type'    => 'select',
                'class'   => 'wc-enhanced-select',
                'default' => 'taxable',
                'options' => array(
                    'taxable' => __( 'Taxable', 'woocommerce' ),
                    'none'    => _x( 'None', 'Tax status', 'woocommerce' ),
                ),
            ),
            'free_shipping_from' => [
                'title' => __('Darmowa wysyłka od kwoty', $namespace),
                'type' => 'number',
                'custom_attributes' => [
                    'step' => 'any',
                    'min' => '0'
                ],
                'default' => '',
                'placeholder' => __('Wyłączona', $namespace)
            ],
        );

        $shipping_classes = WC()->shipping()->get_shipping_classes();
        if (!empty($shipping_classes)) {
            $settings['class_costs'] = array(
                'title' => __('Shipping class costs', 'woocommerce'),
                'type' => 'title',
                'default' => '',
                /* translators: %s: URL for link. */
                'description' => sprintf(__('These costs can optionally be added based on the <a href="%s">product shipping class</a>.', 'woocommerce'), admin_url('admin.php?page=wc-settings&tab=shipping&section=classes')),
            );
            foreach ($shipping_classes as $shipping_class) {
                if (!isset($shipping_class->term_id)) {
                    continue;
                }
                $settings['class_cost_' . $shipping_class->term_id] = array(
                    'title' => sprintf(__('"%s" shipping class cost', 'woocommerce'), esc_html($shipping_class->name)),
                    'type' => 'text',
                    'placeholder' => __('N/A', 'woocommerce'),
                    'description' => $cost_desc,
                    'default' => $this->get_option('class_cost_' . $shipping_class->slug), // Before 2.5.0, we used slug here which caused issues with long setting names.
                    'desc_tip' => true,
                    'sanitize_callback' => array($this, 'sanitize_cost'),
                );
            }

            $settings['no_class_cost'] = array(
                'title' => __('No shipping class cost', 'woocommerce'),
                'type' => 'text',
                'placeholder' => __('N/A', 'woocommerce'),
                'description' => $cost_desc,
                'default' => '',
                'desc_tip' => true,
                'sanitize_callback' => array($this, 'sanitize_cost'),
            );

            $settings['type'] = array(
                'title' => __('Calculation type', 'woocommerce'),
                'type' => 'select',
                'class' => 'wc-enhanced-select',
                'default' => 'class',
                'options' => array(
                    'class' => __('Per class: Charge shipping for each shipping class individually', 'woocommerce'),
                    'order' => __('Per order: Charge shipping for the most expensive shipping class', 'woocommerce'),
                ),
            );
        }

        $this->instance_form_fields = $settings;
    }

    /**
     * @return mixed
     */
    protected function getPostData(string $key)
    {
        if (isset($_POST['post_data'])) {
            $data = [];
            parse_str($_POST['post_data'], $data);
            if (isset($data[$key])) {
                return $data[$key];
            }
        }
        return '';
    }

    public function render(): void
    {
        $tokenManager = new MapsTokenManager();
        $isCod = WC()->session->get('chosen_payment_method') === 'cod';
        $wasSelected = $this->getPostData('polkurier_method_name') === $this->id;

        $selectedPointCod = (bool)$this->getPostData('polkurier_point_cod');
        $selectedPointId = (string)$this->getPostData('polkurier_point_id');
        $selectedPointLabel = (string)$this->getPostData('polkurier_point_label');
        $selectedPointType = (string)$this->getPostData('polkurier_point_type');

        if (!$wasSelected || ($isCod && !$selectedPointCod)) {
            $selectedPointCod = false;
            $selectedPointId = '';
            $selectedPointLabel = '';
        }

        $position = get_option('polkurier_layout_checkout_point_button_position');
        if (!empty(CheckoutPointButtonPosition::MAP_TEMPLATE[$position])) {
            $template = CheckoutPointButtonPosition::MAP_TEMPLATE[$position];
        } else {
            $template = 'checkout/select_shipping_point.php';
        }

        echo View::render($template, [
            'searchAddress' => '',
            'selectedPointId' => $selectedPointId,
            'selectedPointLabel' => $selectedPointLabel,
            'selectedPointCod' => $selectedPointCod,
            'selectedPointType' => $selectedPointType,
            'apiToken' => (string)$tokenManager->getToken(),
            'providerName' => $this->getPolkurierId(),
            'isCod' => $isCod,
            'methodId' => $this->getId(),
            'buttonLabel' => $this->getSelectButtonLabel(),
        ]);
    }

    /**
     * {@inheritDoc}
     * @param array $package
     */
    public function calculate_shipping($package = [])
    {
        $rate = [
            'id' => $this->get_rate_id(),
            'label' => $this->title,
            'cost' => 0,
            'package' => $package,
        ];

        // Free shipping
        if ($this->freeShippingFrom > 0 && $this->getOrderGrossValue($package) >= $this->freeShippingFrom) {
            $this->add_rate($rate);
            return;
        }

        // Calculate the costs.
        $has_costs = false; // True when a cost is set. False if all costs are blank strings.
        $cost = $this->get_option('cost');

        if ('' !== $cost) {
            $has_costs = true;
            $rate['cost'] = $this->evaluate_cost(
                $cost,
                array(
                    'qty' => $this->get_package_item_qty($package),
                    'cost' => $package['contents_cost'],
                )
            );
        }

        // Add shipping class costs.
        $shipping_classes = WC()->shipping()->get_shipping_classes();
        if (!empty($shipping_classes)) {
            $found_shipping_classes = $this->find_shipping_classes($package);
            $highest_class_cost = 0;

            foreach ($found_shipping_classes as $shipping_class => $products) {
                // Also handles BW compatibility when slugs were used instead of ids.
                $shipping_class_term = get_term_by('slug', $shipping_class, 'product_shipping_class');
                $class_cost_string = $shipping_class_term && $shipping_class_term->term_id ? $this->get_option('class_cost_' . $shipping_class_term->term_id, $this->get_option('class_cost_' . $shipping_class, '')) : $this->get_option('no_class_cost', '');

                if ('' === $class_cost_string) {
                    continue;
                }

                $has_costs = true;
                $class_cost = $this->evaluate_cost(
                    $class_cost_string,
                    array(
                        'qty' => array_sum(wp_list_pluck($products, 'quantity')),
                        'cost' => array_sum(wp_list_pluck($products, 'line_total')),
                    )
                );

                if ('class' === $this->type) {
                    $rate['cost'] += $class_cost;
                } else {
                    $highest_class_cost = $class_cost > $highest_class_cost ? $class_cost : $highest_class_cost;
                }
            }

            if ('order' === $this->type && $highest_class_cost) {
                $rate['cost'] += $highest_class_cost;
            }
        }

        // Adds COD additional fee
        if (WC()->session->get('chosen_payment_method') === 'cod') {
            $rate['cost'] += (float)Numbers::ensureDigits($this->get_option('cost_cod'));
        }

        if ($has_costs) {
            $this->add_rate($rate);
        }

        /**
         * Developers can add additional flat rates based on this one via this action since @version 2.4.
         *
         * Previously there were (overly complex) options to add additional rates however this was not user.
         * friendly and goes against what Flat Rate Shipping was originally intended for.
         */
        do_action('woocommerce_' . $this->id . '_shipping_add_rate', $this, $rate);
    }

    /**
     * Get items in package.
     *
     * @param array $package Package of items from cart.
     * @return int
     */
    public function get_package_item_qty($package)
    {
        $total_quantity = 0;
        foreach ($package['contents'] as $item_id => $values) {
            if ($values['quantity'] > 0 && $values['data']->needs_shipping()) {
                $total_quantity += $values['quantity'];
            }
        }
        return $total_quantity;
    }

    /**
     * Evaluate a cost from a sum/string.
     *
     * @param string $sum Sum of shipping.
     * @param array $args Args, must contain `cost` and `qty` keys. Having `array()` as default is for back compat reasons.
     * @return string
     */
    protected function evaluate_cost($sum, $args = array())
    {
        // Add warning for subclasses.
        if (!is_array($args) || !array_key_exists('qty', $args) || !array_key_exists('cost', $args)) {
            wc_doing_it_wrong(__FUNCTION__, '$args must contain `cost` and `qty` keys.', '4.0.1');
        }

        include_once WC()->plugin_path() . '/includes/libraries/class-wc-eval-math.php';

        // Allow 3rd parties to process shipping cost arguments.
        $args = apply_filters('woocommerce_evaluate_shipping_cost_args', $args, $sum, $this);
        $locale = localeconv();
        $decimals = array(wc_get_price_decimal_separator(), $locale['decimal_point'], $locale['mon_decimal_point'], ',');
        $this->fee_cost = $args['cost'];

        // Expand shortcodes.
        add_shortcode('fee', array($this, 'fee'));

        $sum = do_shortcode(
            str_replace(
                array(
                    '[qty]',
                    '[cost]',
                ),
                array(
                    $args['qty'],
                    $args['cost'],
                ),
                $sum
            )
        );

        remove_shortcode('fee', array($this, 'fee'));

        // Remove whitespace from string.
        $sum = preg_replace('/\s+/', '', $sum);

        // Remove locale from string.
        $sum = str_replace($decimals, '.', $sum);

        // Trim invalid start/end characters.
        $sum = rtrim(ltrim($sum, "\t\n\r\0\x0B+*/"), "\t\n\r\0\x0B+-*/");

        // Do the math.
        return $sum ? \WC_Eval_Math::evaluate($sum) : 0;
    }

    /**
     * Work out fee (shortcode).
     *
     * @param array $atts Attributes.
     * @return string
     */
    public function fee($atts)
    {
        $atts = shortcode_atts(
            array(
                'percent' => '',
                'min_fee' => '',
                'max_fee' => '',
            ),
            $atts,
            'fee'
        );

        $calculated_fee = 0;

        if ($atts['percent']) {
            $calculated_fee = $this->fee_cost * (floatval($atts['percent']) / 100);
        }

        if ($atts['min_fee'] && $calculated_fee < $atts['min_fee']) {
            $calculated_fee = $atts['min_fee'];
        }

        if ($atts['max_fee'] && $calculated_fee > $atts['max_fee']) {
            $calculated_fee = $atts['max_fee'];
        }

        return $calculated_fee;
    }

    /**
     * Finds and returns shipping classes and the products with said class.
     *
     * @param mixed $package Package of items from cart.
     * @return array
     */
    public function find_shipping_classes($package)
    {
        $found_shipping_classes = array();
        foreach ($package['contents'] as $item_id => $values) {
            if ($values['data']->needs_shipping()) {
                $found_class = $values['data']->get_shipping_class();

                if (!isset($found_shipping_classes[$found_class])) {
                    $found_shipping_classes[$found_class] = array();
                }

                $found_shipping_classes[$found_class][$item_id] = $values;
            }
        }
        return $found_shipping_classes;
    }

    /**
     * @param array $package
     * @return float
     */
    private function getOrderGrossValue($package = [])
    {
        $value = 0;
        foreach ($package['contents'] as $item) {
            $value += $item['line_total'] + $item['line_tax'];
        }
        return $value;
    }

    public function onCheckoutProcess(?WC_Order $order = null): void
    {
        if ($this->isSelected()) {
            if ($order !== null && empty($order->get_billing_phone())) {
                throw new \InvalidArgumentException($this->getErrorMessage(self::ERROR_MESSAGE_NO_PHONE_NUMBER));
            }

            $pointId = null;
            if (!empty($_POST['polkurier_point_id'])) {
                $pointId = $_POST['polkurier_point_id'];
            }
            if (empty($pointId)) {
                $sessionStore = WC()->session->get(Polkurier::NAME . '_delivery_point');
                if (!empty($sessionStore['id'])) {
                    $pointId = $sessionStore['id'];
                }
            }
            if (empty($pointId) && WC()->cart->needs_shipping()) {
                throw new \InvalidArgumentException($this->getErrorMessage(self::ERROR_MESSAGE_NOT_POINT_SELECTED));
            }
        }
    }

    public function onCheckoutUpdateOrderReview(): void
    {
        if ($this->isSelected()) {
            // Czyści cache stawek za przesyłkę (aby przeliczyć stawki po zmianie sposobu płatności)
            $packages = WC()->cart->get_shipping_packages();
            foreach ($packages as $key => $value) {
                unset(WC()->session->{"shipping_for_package_$key"});
            }
        }
    }

    /**
     * Sanitize the cost field.
     *
     * @param string $value Unsanitized value.
     * @return string
     * @throws \Exception Last error triggered.
     * @since 3.4.0
     */
    public function sanitize_cost($value)
    {
        $value = is_null($value) ? '' : $value;
        $value = wp_kses_post(trim(wp_unslash($value)));
        $value = str_replace(array(get_woocommerce_currency_symbol(), html_entity_decode(get_woocommerce_currency_symbol())), '', $value);
        // Thrown an error on the front end if the evaluate_cost will fail.
        $dummy_cost = $this->evaluate_cost(
            $value,
            array(
                'cost' => 1,
                'qty' => 1,
            )
        );
        if (false === $dummy_cost) {
            throw new \Exception(\WC_Eval_Math::$last_error);
        }
        return $value;
    }

}
