import {html, css, LitElement} from "../esm/libs/lit.dist.js?ver=2.5.1";
import styles from '../esm/custom-elements/styles.css.js?ver=2.5.1';
import {openMapModal} from "../esm/util/map.js?ver=2.5.1";
import {getPointLabel, isSenderPointRequired} from "../esm/util/couriers.js?ver=2.5.1";
import '../esm/custom-elements/pk-badge.js?ver=2.5.1';
import '../esm/custom-elements/pk-card.js?ver=2.5.1';
import '../esm/custom-elements/pk-icon.js?ver=2.5.1';
import '../esm/custom-elements/pk-select.js?ver=2.5.1';

export class PkOrderPickupView extends LitElement {

    static properties = {
        courier: {type: String},
        pickup: {type: Object},
        sender: {type: Object},
        pickupPoint: {type: Object},
        pickupDates: {type: Array},
        hasNoCourierOrder: {type: Boolean},
        hasPickupDate: {type: Boolean},
        hasPickupTime: {type: Boolean},
    };

    static styles = [
        styles,
        css`
            :host {
                display: flex;
                flex-direction: column;
                gap: 0.25rem;
            }
            .row {
                display: flex;
                gap: 0.5rem;
                align-items: baseline;
            }
        `,
    ];

    setPickup(props) {
        this.pickup = {
            ...(this.pickup || {}),
            ...props,
        };
        this.dispatchEvent(new CustomEvent('change', {
            detail: this.pickup,
            bubbles: true,
            composed: true,
        }));
    }

    setSender(props) {
        this.sender = {
            ...(this.sender || {}),
            ...props,
        };
        this.dispatchEvent(new CustomEvent('change'));
    }

    get pickupTimes() {
        if (this.pickupDates?.length && this.pickup?.pickupdate) {
            for (let date of this.pickupDates) {
                if (date.pickupdate === this.pickup.pickupdate) {
                    return date.time || [];
                }
            }
        }
        return [];
    }

    async onClickSelectPointButton() {
        const point = await openMapModal({
            selectedProviders: this.courier ? [this.courier] : undefined,
            searchQuery: this.sender?.point_id || this.sender?.postcode || '',
            functions: ['send'],
        });
        if (point) {
            this.setSender({
                point_id: point.id,
                delivery_point: {
                    code: point.id,
                    provider: point.provider,
                    label: point.address,
                },
            });
        }
    }

    render() {
        return html`
            <div class="row">
                <input
                    type="radio"
                    id="pk-pickup-nocourierorder"
                    .checked="${!this.pickup?.nocourierorder}"
                    .disabled="${!this.hasPickupDate}"
                    @change="${() => this.setPickup({
                        nocourierorder: false,
                    })}"
                >
                <div style="flex-grow: 1">
                    <label for="pk-pickup-nocourierorder">
                        Zamawiam kuriera
                    </label>
                    ${this.hasPickupDate && !this.pickup?.nocourierorder ? html`
                        <div class="row">
                            <pk-select
                                .options="${this.pickupDates}"
                                .value="${this.pickup?.pickupdate}"
                                labels="pickupdate"
                                values="pickupdate"
                                @select="${e => this.setPickup({
                                    pickupdate: e.target.value,
                                    pickuptimefrom: null,
                                    pickuptimeto: null,
                                })}"
                                style="flex-grow: 1"
                            ></pk-select>
                            ${this.hasPickupTime ? html`
                                <pk-select
                                    .options="${this.pickupTimes}"
                                    .value="${`${this.pickup?.pickuptimefrom}-${this.pickup?.pickuptimeto}`}"
                                    .labels="${el => `${el.timefrom}-${el.timeto}`}"
                                    .values="${el => `${el.timefrom}-${el.timeto}`}"
                                    @select="${e => {
                                        const [pickuptimefrom, pickuptimeto] = String(e.target.value).split('-');
                                        this.setPickup({
                                            pickuptimefrom,
                                            pickuptimeto,
                                        });
                                    }}"
                                ></pk-select>
                            ` : null}
                        </div>
                    ` : null}
                </div>
            </div>
            <div class="row">
                <input
                    type="radio"
                    id="pk-pickup-courierorder"
                    .checked="${this.pickup?.nocourierorder === true}"
                    .disabled="${!this.hasNoCourierOrder}"
                    @change="${() => this.setPickup({
                        nocourierorder: true,
                        pickupdate: null,
                        pickuptimefrom: null,
                        pickuptimeto: null,
                        multiPickup: false,
                    })}"
                >
                <div style="flex-grow: 1">
                    <label for="pk-pickup-courierorder">
                        ${getPointLabel(this.courier)}
                    </label>
                    ${this.pickup?.nocourierorder && isSenderPointRequired(this.courier) ? html`
                        <div class="d-flex gap-sm mt-xs" type="Odbiór w punkcie">
                            <div>
                                <pk-icon icon="location-pin" size="xs"></pk-icon>
                            </div>
                            <div>
                                <div class="d-flex gap-sm align-items-center">
                                    <pk-badge type="code" size="sm" class="fw-bold" style="margin-left: -0.25rem;">
                                        ${this.pickupPoint?.code || 'Wybierz punkt nadania'}
                                    </pk-badge>
                                    <button class="btn btn-raised btn-rounded btn-icon" @click="${this.onClickSelectPointButton}" title="Zmień punkt nadania">
                                        <pk-icon icon="${this.pickupPoint?.code ? 'pencil' : 'plus'}" size="text" type="plain" ></pk-icon>
                                    </button>
                                </div>
                                ${this.pickupPoint?.label ? html`
                                    <div class="fs-nano text-muted">${this.pickupPoint.label}</div>
                                ` : null}
                            </div>
                        </div>
                    ` : null}
                </div>
            </div>
        `;
    }

}

customElements.define("pk-order-pickup-view", PkOrderPickupView);
