import {createRef, html, LitElement, ref} from "../libs/lit.dist.js?ver=2.5.1";
import styles from './pk-alert-dialog.css.js?ver=2.5.1';

/**
 * The `PkAlertDialog` class represents a customizable alert dialog component designed to render modal dialogs
 * with configurable titles, content, and actions. This component supports both acceptance and cancellation events
 * with lifecycle management and visual transitions.
 *
 * Key Features:
 * - Dynamically open and close the alert dialog using `open()` and `close()` methods.
 * - Tracks the state of the dialog through the `opened` property that triggers transition events when changed.
 * - Includes built-in customizable event handling for accept and cancel actions.
 * - Allows the use of light DOM slots for title, content, and footer sections within the dialog.
 * - Fires `opened` and `closed` events to notify when the dialog's visibility state changes due to transitions.
 */
export class PkAlertDialog extends LitElement {

    #dialogRef = createRef();
    static styles = styles;

    static properties = {
        opened: {type: Boolean},
        title: {type: String},
    };

    get dialog() {
        return this.#dialogRef?.value;
    }

    open() {
        this.opened = true;
    }

    close() {
        this.opened = false;
    }

    dispatchViewEvent(eventName, detail = {}, options = {}) {
        const event = new CustomEvent(eventName, {
            cancelable: true,
            bubbles: true,
            composed: true,
            ...options,
            detail,
        });
        this.dispatchEvent(event);
        return event;
    }

    updated(changedProps) {
        if (changedProps.has('opened')) {
            if (this.opened) {
                this.dialog?.showModal();
            } else {
                this.dialog?.close();
            }
        }
    }

    #onClickHandler(e) {
        const button = e.target.closest('button');
        if (button?.dataset.cancel !== undefined) {
            e.preventDefault();
            e.stopPropagation();
            const event = this.dispatchViewEvent('reject');
            if (!event.defaultPrevented) {
                this.opened = false;
            }
            return;
        }

        if (button?.dataset.accept !== undefined) {
            e.preventDefault();
            e.stopPropagation();
            const event = this.dispatchViewEvent('accept');
            if (!event.defaultPrevented) {
                this.opened = false;
            }
        }
    }

    #onTransitionEnd(event) {
        if (event.propertyName === 'opacity' && !this.opened) {
            this.dispatchEvent(new Event('closed'));
        }
        if (event.propertyName === 'opacity' && this.opened) {
            this.dispatchEvent(new Event('opened'));
        }
    }

    render() {
        return html`
            <dialog
                    ${ref(this.#dialogRef)}
                    @click="${this.#onClickHandler}"
                    @close="${() => {
                        this.opened = false;
                        this.dispatchViewEvent('close');
                    }}"
                    @transitionend="${this.#onTransitionEnd}"
                    @transitioncancel="${this.#onTransitionEnd}"
            >
                <slot name="icon"></slot>
                <slot name="title">${this.title}</slot>
                <slot></slot>
                <div>
                    <slot name="footer"></slot>
                </div>
            </dialog>
        `;
    }
}

customElements.define('pk-alert-dialog', PkAlertDialog);

let nextButtonId = 1;

/**
 * @param {string} title
 * @param {string} message
 * @param {Array<{
 *      primary?: boolean,
 *      destructive?: boolean,
 *      label: string,
 *      value: string|number|boolean|undefined,
 *      class?: string,
 *      onClick?: function,
 *      close?: boolean,
 *  }>} buttons
 * @param {{
 *     className?: string,
*      icon?: string,
 * }} options
 * @return {Promise<boolean|string|undefined>}
 */
export function createAlert(title = '', message = '', buttons = [], options = {}) {
    const buttonsById = {};
    const dialog = document.createElement('pk-alert-dialog');
    dialog.className = options.className || '';

    dialog.innerHTML = `
        ${options.icon ? `<i class="${options.icon}" slot="icon"></i>` : ''}
        <div slot="title">${title || ''}</div>
        ${message || ''}
    `;

    if (Array.isArray(buttons)) {
        let buttonId;
        for (let button of buttons) {
            buttonId = `pk-alert-dialog-button-${nextButtonId++}`;
            buttonsById[buttonId] = button;
            dialog.innerHTML += `
                    <button 
                        type="button" 
                        slot="footer" 
                        class="${button.destructive ? 'destructive' : ''} ${button.primary ? 'primary' : ''} ${button.class || ''} waves-effect" 
                        id="${buttonId}"
                        data-value="${button.value || ''}"
                    >${button.label || ''}</button>
                `;
        }
    }

    let resolvePromise;
    const promise = new Promise((resolve) => {
        resolvePromise = resolve;
    });

    dialog.addEventListener('click', (e) => {
        const buttonId = e.target.closest('button')?.id;
        const button = buttonsById[buttonId];
        if (button) {
            if (typeof button.onClick === 'function') {
                button.onClick(button.value, dialog);
            }
            resolvePromise(button.value, dialog);
            if (button.close !== false) {
                dialog.opened = false;
            }
        }
    });

    dialog.addEventListener('close', () => {
        setTimeout(() => {
            document.body.removeChild(dialog);
        }, 400)
    })

    document.body.appendChild(dialog);
    dialog.opened = true;
    return promise;
}

/**
 * @param {string} title
 * @param {string} message
 * @param {{
 *      destructive?: boolean,
 *      acceptLabel?: string,
 *      cancelLabel?: string
 *      className?: string,
 *      icon?: string,
 *  }} options
 * @return {Promise<boolean|string|undefined>}
 */
export function createConfirm(title = '', message = '', {destructive, acceptLabel, cancelLabel, className, icon} = {}) {
    return createAlert(title, message, [{
        destructive,
        primary: !destructive,
        label: acceptLabel || 'Tak',
        value: true,
    }, {
        label: cancelLabel || 'Anuluj',
        value: false,
    }], {
        className,
        icon,
    });
}

export const deleteConfirmOptions = {
    destructive: true,
    acceptLabel: 'Usuń',
};
