import {html, css, LitElement, unsafeHTML} from "../libs/lit.dist.js?ver=2.5.1";

export const objectToOptions = (object, defaults = {}) => Object.keys(object).map(key => ({
    ...defaults,
    value: key,
    label:  object[key] || '',
}));

const groupBy = (collection, groupBy) => {
    const groups = {};
    let key;
    for (let item of collection) {
        key = item[groupBy] || '';
        if (!groups[key]) {
            groups[key] = [];
        }
        groups[key].push(item);
    }
    return groups;
}

export class PkSelect extends LitElement {

    static properties = {
        options: {type: Array},
        name: {type: String},
        class: {type: String},
        className: {type: String},
        tabindex: {},
        value: {},
        labels: {},
        values: {},
        groups: {},
        titles: {},
        empty: {},
        disabled: {type: Boolean},
        multiple: {type: Boolean},
    };

    static styles = css`
        :host {
            display: block;
            width: 100%;
        }
        
        select {
            border-radius: var(--pk-input-border-radius);
            padding: var(--pk-input-padding);
            display: block;
            width: 100%;
            border: 1px solid var(--pk-input-border-color);
            color: var(--pk-input-color);
            background: var(--pk-input-background);
            
            &:focus {
                border-color: var(--pk-input-active-color);
                box-shadow: 0 0 0 .25rem color(from var(--pk-input-active-color) xyz calc(x * 1.3) calc(y * 1.3) calc(z * 1.3) / 0.3)
            }
            
            &[disabled] {
                background: var(--pk-input-disabled-background);
            }
        }
    `;

    constructor() {
        super();
        this.options = [];
        this.name = '';
        this.class = '';
        this.className = '';
        this.value = '';
        this.labels = 'label';
        this.values = 'value';
        this.titles = 'title';
        this.groups = '';
        this.empty = '---';
        this.disabled = false;
        this.multiple = false;
        this.tabindex = 0;
    }

    get selectedOptions() {
        let value;
        if (this.multiple) {
            value = this.value;
        } else {
            value = [this.value];
        }
        return this.getOptions()
            .filter(option => value.includes(value, String(option.value)))
            .map(option => option.option);
    }

    getOptions() {
        let options = this.options?.map(option => ({
            label: this.getOptionLabel(option) || '',
            value: this.getOptionValue(option) || '',
            group: this.getOptionGroup(option) || '',
            title: this.getOptionTitle(option) || '',
            disabled: option.disabled === true,
            option,
        })) || [];

        if (this.empty) {
            let empty = {};
            if (typeof this.empty === 'object') {
                empty = {
                    label: '',
                    value: '',
                    group: '',
                    ...this.empty
                };
            } else {
                empty = {
                    label: this.empty,
                    value: '',
                    group: '',
                };
            }
            options = [empty, ...options];
        }
        return options;
    }

    getOptionLabel(option) {
        if (this.labels && typeof this.labels === 'function') {
            return this.labels(option);
        }
        return option[this.labels];
    }

    getOptionValue(option) {
        if (this.values && typeof this.values === 'function') {
            return this.values(option);
        }
        return option[this.values];
    }

    getOptionGroup(option) {
        if (this.groups && typeof this.groups === 'function') {
            return this.groups(option);
        }
        return option[this.groups];
    }

    getOptionTitle(option) {
        if (this.titles && typeof this.titles === 'function') {
            return this.titles(option);
        }
        return option[this.titles];
    }

    isSelected(option) {
        if (Array.isArray(this.value)) {
            return this.value.includes(option.value);
        }
        return option.value == this.value;
    }

    onChangeHandler(event) {
        const selectedValues = [];
        for (let option of event.target.selectedOptions) {
            selectedValues.push(option.value);
        }

        if (this.multiple) {
            this.value = selectedValues;
        } else {
            this.value = selectedValues[0];
        }

        this.dispatchEvent(new CustomEvent('select', {
            bubbles: true,
            composed: true,
        }));
    }

    renderOptgroup(groupName, options) {
        const renderedOptions = this.renderSelectOptions(options);
        if (groupName.length > 0) {
            return html`<optgroup .label=${groupName}>${renderedOptions}</optgroup>`;
        }
        return renderedOptions;
    }

    renderSelectOptions(options) {
        return options.map(option => html`
            <option .value=${option.value} .disabled=${option.disabled} .title="${option.title}" .selected=${this.isSelected(option)}>
                ${unsafeHTML(option.label)}
            </option>
        `);
    }

    render() {
        const options = this.getOptions();
        const groupedOptions = groupBy(options, 'group');
        const groups = Object.keys(groupedOptions);
        return html`
            <select 
                class="${this.className}" 
                .name=${this.name} 
                .style="${this.style}"
                @input=${e => this.onChangeHandler(e)} 
                ?multiple=${this.multiple} 
                ?disabled=${this.disabled}
                tabindex="${this.tabindex || 0}"
            >
                ${groups.map(group => this.renderOptgroup(group, groupedOptions[group]))}
            </select>
        `;
    }

}

customElements.define('pk-select', PkSelect);
