export const GET = 'GET';
export const POST = 'POST';
export const PUT = 'PUT';
export const DELETE = 'DELETE';

export const objectToUrlEncoded = (source, target = {}, path = '') => {
    let currentPath;
    if (Array.isArray(source)) {
        let index = 0;
        for (let item of source) {
            currentPath = `${path}[${index++}]`;
            if (typeof item === 'object') {
                objectToUrlEncoded(item, target, currentPath);
            } else {
                target[currentPath] = item;
            }
        }
    } else if (typeof source === 'object') {
        for (let key in source) {
            if (path) {
                currentPath = `${path}[${key}]`;
            } else {
                currentPath = key;
            }
            if (typeof source[key] === 'object') {
                objectToUrlEncoded(source[key], target, currentPath);
            } else if (source[key]) {
                target[currentPath] = source[key];
            }
        }
    }
    return new URLSearchParams(target);
}

/**
 * Wykonuje zapytanie REST
 * @param {string} method (GET|POST|PUT|DELETE)
 * @param {string} url
 * @param {object|null} data
 * @param {object} options
 * @return {Promise<any>}
 */
export const httpRequest = async (method, url, data= null, options = {}) => {

    const requestOptions = {
        method,
        mode: options.mode || 'cors',
        cache: 'no-cache',
        redirect: options.redirect || 'follow',
        credentials: options.credentials || 'same-origin',
        referrerPolicy: options.referrerPolicy || 'no-referrer-when-downgrade',
        headers: {
            'Content-Type': 'application/json',
            ...(options.headers || {}),
        },
    };

    if (method === POST || method === PUT) {
        if (requestOptions.headers['Content-Type'] === 'application/json') {
            requestOptions.body = JSON.stringify(data);
        } else if (requestOptions.headers['Content-Type'] === 'application/x-www-form-urlencoded') {
            if (data instanceof FormData) {
                requestOptions.body = data;
            } else {
                requestOptions.body = objectToUrlEncoded(data);
            }
        }
    }

    const response = await fetch(url, requestOptions);
    const contentType = response.headers.get('content-type');
    response.responseBody = await response.text();
    response.responseJSON = response.responseBody && contentType === 'application/json'
        ? JSON.parse(response.responseBody)
        : null;

    if (!response.ok) {
        throw response;
    }

    return contentType === 'application/json'
        ? response.responseJSON
        : response.responseBody;
};

export const ajaxRequest = (endpoint, data = {}, options = {}) => httpRequest(POST, POLKURIER.ajaxUrl, {
    _ajax_nonce: POLKURIER.ajaxNonce,
    action: endpoint,
    ...data,
}, {
    ...options,
    headers: {
        ...(options.headers || {}),
        "Content-Type": "application/x-www-form-urlencoded",
    },
});
