<?php

namespace Polkurier;

/**
 * @template T
 */
abstract class AbstractSimpleObjectManager
{

    /**
     * @var T[]
     */
    private static array $items;

    public static function generateId(): string
    {
        return wp_generate_uuid4();
    }

    abstract protected function getWPOptionName(): string;

    /**
     * @return T
     */
    abstract public function fromArray(array $data): object;

    private function isItemsStorageInitialized(): bool
    {
        return isset(self::$items[$this->getWPOptionName()]);
    }

    /**
     * @return T[]
     */
    private function getItemsStorage(): array
    {
        return self::$items[$this->getWPOptionName()] ?? [];
    }

    /**
     * @param T $item
     */
    private function addToItemsStorage($item): void
    {
        self::$items[$this->getWPOptionName()][] = $item;
    }

    /**
     * @param T[] $items
     */
    private function setItemsStorage(array $items): void
    {
        self::$items[$this->getWPOptionName()] = $items;
    }

    /**
     * @param string $id
     * @return T|null
     */
    public function getById(string $id): ?object
    {
        foreach ($this->getAll() as $item) {
            if ($item->id === $id) {
                return $item;
            }
        }
        return null;
    }

    /**
     * @return T[]
     */
    public function getAll(): array
    {
        if (!$this->isItemsStorageInitialized()) {
            $data = (array)json_decode(get_option($this->getWPOptionName()), true);
            foreach ($data as $row) {
                $this->addToItemsStorage($this->fromArray($row));
            }
        }
        return $this->getItemsStorage();
    }

    /**
     * @param T $item
     */
    public function persist($item): self
    {
        if (!$item->id) {
            $this->addToItemsStorage($item);
        } else {
            $previousItem = $this->getById($item->id);
            if ($previousItem === null) {
                $this->addToItemsStorage($item);
            }
        }
        return $this;
    }

    /**
     * @param T $item
     */
    public function remove($item): self
    {
        $this->setItemsStorage(array_filter($this->getAll(), static function ($storedItem) use ($item) {
            return $storedItem !== $item;
        }));
        return $this;
    }

    /**
     * @param T[] $items
     * @return array[]
     */
    protected function normalizeItems(array $items): array
    {
        return $items;
    }

    public function save(): self
    {
        foreach ($this->getAll() as $item) {
            if ($item->id === null) {
                $item->id = self::generateId();
            }
        }
        update_option($this->getWPOptionName(), json_encode($this->normalizeItems($this->getAll())));
        return $this;
    }

}
