<?php

namespace Polkurier;

use Polkurier\Exception\ApiException;
use Polkurier\Types\ConfigType;

class ApiClient
{

    protected string $apiLogin;
    protected string $apiToken;
    private ConfigManager $config;

    public function __construct(string $login = '', string $token = '')
    {
        $this->config = new ConfigManager();

        if (empty($login)) {
            $login = $this->config->getString(ConfigType::ACCOUNT_ID);
        }

        if (empty($token)) {
            $token = $this->config->getString(ConfigType::APIKEY);
        }

        $this->apiLogin = $login;
        $this->apiToken = $token;
    }

    /**
     * @throws ApiException
     */
    protected function call(string $method, array $params = []): array
    {
        global $wp_version;

        $params['platform'] = 2;
        $params['platform_version'] = Polkurier::VERSION;
        $params['platform_info'] = [
            'version' => Polkurier::VERSION,
            'PHP_version' => PHP_VERSION,
            'WP_version' => $wp_version,
            'WC_version' => WC()->version,
            'IID' => $this->config->getString(ConfigType::INSTALLATION_ID),
        ];

        [$httpCode, $response] = $this->httpRequest(json_encode([
            'authorization' => [
                'login' => $this->apiLogin,
                'token' => $this->apiToken,
            ],
            'apimetod' => $method,
            'data' => $params,
        ]));

        if ($httpCode !== 200) {
            throw new ApiException("Błąd połączenia z API serwisu POLKURIER.PL. CODE $httpCode (E_NO_CONNECTION)");
        }

        $arrayResult = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new ApiException('Błąd połączenia z API POLKURIER.PL. (E_INVALID_JSON)');
        }

        if (!is_array($arrayResult)) {
            throw new ApiException('Błąd połączenia z API POLKURIER.PL. (E_EMPTY_RESPONSE)');
        }

        if (($arrayResult['status'] ?? '') === 'error') {
            throw new ApiException($arrayResult['response']);
        }

        return $arrayResult;
    }

    /**
     * @return array [int, string]
     */
    private function httpRequest(string $body): array
    {
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $this->config->getString(ConfigType::API_URL));
        curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'POST');
        curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Content-Length: ' . strlen($body),
        ]);
        $response = (string)curl_exec($curl);
        $httpCode = (int)curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);
        return [$httpCode, $response];
    }

    /**
     * Wykonuje zapytanie do API
     * @return mixed
     * @throws ApiException
     */
    public function makeApiRequest(string $method, array $params = [])
    {
        return $this->call($method, $params)['response'];
    }

}
