<?php

namespace Polkurier\Controller;


use Exception;
use Polkurier\AddressBook\AddressBookManager;
use Polkurier\ApiClient;
use Polkurier\Exception\HttpException;
use Polkurier\Http\JsonResponse;
use Polkurier\Http\Request;
use Polkurier\Order\Order;
use Polkurier\Order\OrderAddress;
use Polkurier\Order\OrderCod;
use Polkurier\Order\OrderCreator;
use Polkurier\Order\OrderPack;
use Polkurier\Order\OrderPickup;
use Polkurier\Util\Arr;
use Polkurier\Util\Numbers;
use Polkurier\Util\ParameterBag;
use Throwable;

class OrderController
{

    /**
     * Wycena zamówienia
     * @throws HttpException
     * @depecated
     */
    public function evaluateAction(Request $request): JsonResponse
    {

        $params = [
            "shipmenttype" => $request->request->getString('polkurier_type'),
            "packs" => [],
            'COD' => $request->request->getFloat('polkurier_pobranie'),
            'codtype' => $request->request->getString('polkurier_pobranie_zwrot'),
            'return_cod' => $request->request->getString('polkurier_pobranie_rodzaj'),
            'insurance' => $request->request->getFloat('polkurier_ubezpieczenie'),
            'postcode_recipient' => '',
            'postcode_sender' => '',
            'recipient_country' => '',
            'courierservice' => $request->request->getArray('courierservice'),
        ];

        $packagesList = json_decode(stripslashes($request->request->getString('packages_list', '[]')), true);
        foreach ($packagesList as $package) {
            $params['packs'][] = [
                "length" => (float)$package['depth'],
                "width" => (float)$package['width'],
                "height" => (float)$package['height'],
                "weight" => (float)$package['weight'],
                "amount" => (int)$package['count'],
                "type" => $package['nonstandard'] ? "NST" : "ST",
            ];
        }

        $orderId = $request->request->getInt('order_id');
        if ($orderId > 0) {
            $order = wc_get_order($orderId);
            if ($order !== false) {
                $orderData = $order->get_data();
                $params['recipient_country'] = $orderData['shipping']['country'];
                $params['postcode_recipient'] = $orderData['shipping']['postcode'];
            }
        }

        $senderAddress = (new AddressBookManager())->getById($request->request->getString('polkurier_address'));
        if ($senderAddress !== null) {
            $params["postcode_sender"] = $senderAddress->postcode;
        }

        if (empty($params['packs']) || empty($params['shipmenttype'])) {
            return new JsonResponse([
                'success' => false,
            ]);
        }

        $recipient = $request->request->getArray('recipient');
        $recipientCountry = Arr::get($recipient, 'country', '');
        if ($recipientCountry !== '') {
            $params['recipient_country'] = $recipientCountry;
        }
        $coverAddress = $request->request->get('polkurier_cover_address');

        if ($coverAddress !== 'null') {
            $params['courierservice'] = [
                'COVER_ADDRESS_SENDER' => 1
            ];
        }

        try {
            $response = (new ApiClient())->makeApiRequest('order_valuation', $params);
            return new JsonResponse([
                'success' => true,
                'data' => $response,
            ]);
        } catch (Exception $e) {
            throw new HttpException($e->getMessage(), 403);
        }
    }

    /**
     * Zapisuje nowe zamówienie
     * @throws HttpException
     * @throws Throwable
     */
    public function createAction(Request $request): JsonResponse
    {
        try {
            $props = $request->request;
            $recipient = new ParameterBag($props->getArray('recipient'));
            $sender = new ParameterBag($props->getArray('sender'));
            $coverAddress = new ParameterBag($props->getArray('cover_address'));
            $pickup = new ParameterBag($props->getArray('pickup'));
            $cod = new ParameterBag($props->getArray('cod'));

            $createAddress = static function (ParameterBag $data) {
                $address = new OrderAddress();
                $address->company = $data->getString('company');
                $address->person = $data->getString('person');
                $address->street = $data->getString('street');
                $address->housenumber = $data->getString('housenumber');
                $address->flatnumber = $data->getString('flatnumber');
                $address->postcode = $data->getString('postcode');
                $address->city = $data->getString('city');
                $address->email = $data->getString('email');
                $address->phone = $data->getString('phone');
                $address->country = $data->getString('country');
                $address->point_id = $data->getString('point_id');
                return $address;
            };

            $order = (new Order())
                ->setShopOrderId($props->getInt('order_id'))
                ->setCourier($props->getString('courier'))
                ->setShipmentType($props->getString('shipmenttype'))
                ->setDescription($props->getString('description'))
                ->setSender($createAddress($sender))
                ->setRecipient($createAddress($recipient))
                ->setCoverAddress($createAddress($coverAddress))
                ->setPacks(array_map(static function ($row) {
                    $pack = new OrderPack();
                    $pack->length = (int)Numbers::ensureDigits((string)($row['length'] ?? '0'));
                    $pack->width = (int)Numbers::ensureDigits((string)($row['width'] ?? '0'));
                    $pack->height = (int)Numbers::ensureDigits((string)($row['height'] ?? '0'));
                    $pack->weight = (float)Numbers::ensureDigits((string)($row['weight'] ?? '0'));
                    $pack->amount = (int)Numbers::ensureDigits((string)($row['amount'] ?? '0'));
                    $pack->type = (string)($row['type'] ?? 'ST');
                    return $pack;
                }, $props->getArray('packs')))
                ->setPickup(new OrderPickup(
                    $pickup->getString('pickupdate'),
                    $pickup->getString('pickuptimefrom'),
                    $pickup->getString('pickuptimeto'),
                    $pickup->getBoolean('nocourierorder')
                ))
                ->setCod(new OrderCod(
                    (float)Numbers::ensureDigits($cod->getString('codamount')),
                    $cod->getString('codtype'),
                    $cod->getString('return_cod'),
                    $cod->getString('codbankaccount')
                ))
                ->setInsurance((float)Numbers::ensureDigits($props->getString('insurance')));

            foreach ($props->getArray('courierservice') as $service => $enabled) {
                if ($enabled === true || $enabled === 'true') {
                    $order->setCourierService($service);
                }
            }

            (new OrderCreator())
                ->setFormVersion($props->getInt('form_version'))
                ->create($order);

            return new JsonResponse([
                'order_id' => $order->getShopOrderId(),
                'order_number' => $order->getOrderNumber(),
                'label' => $order->getLabel(),
                'price_gross' => $order->getPriceGross(),
                'price_net' => $order->getPriceNet(),
            ]);
        } catch (Exception $e) {
            throw new HttpException($e->getMessage(), 403);
        }
    }

    /**
     * Wycena zamówienia
     * @throws HttpException
     */
    public function evaluateV2Action(Request $request): JsonResponse
    {
        $props = $request->request;
        $recipient = new ParameterBag($props->getArray('recipient'));
        $sender = new ParameterBag($props->getArray('sender'));
        $cod = new ParameterBag($props->getArray('cod'));

        $params = [
            "shipmenttype" => $props->get('shipmenttype'),
            'COD' => $cod->getFloat('codamount'),
            'codtype' => $cod->get('codtype', ''),
            'return_cod' => $cod->get('return_cod', ''),
            'insurance' => $props->getFloat('insurance'),
            'postcode_recipient' => $recipient->get('postcode'),
            'recipient_country' => $recipient->get('country'),
            'postcode_sender' => $sender->get('postcode'),
            'sender_country' => $sender->get('country'),
            'courierservice' => $props->getArray('courierservice'),
            'packs' => $props->getArray('packs'),
        ];

        try {
            return new JsonResponse((new ApiClient())->makeApiRequest('order_valuation', $params));
        } catch (Exception $e) {
            throw new HttpException($e->getMessage(), 403);
        }
    }

}
