<?php

namespace Polkurier\Integrations;

use InvalidArgumentException;
use Polkurier\Http\Request;
use Polkurier\MapsTokenManager;
use Polkurier\Polkurier;
use Polkurier\Types\CheckoutPointButtonPosition;
use Polkurier\Util\Arr;
use Polkurier\View;
use WC_Order;
use WC_Shipping_Method;

class FlexibleShipping extends AbstractIntegration
{

    public const ID = 'pk_flexible_shipping';
    public const PROVIDER_INPOST = 'INPOST_PACZKOMAT';
    public const PROVIDER_PACZKA_W_RUCHU = 'PACZKA_W_RUCHU';
    public const PROVIDER_FEDEX = 'FEDEX_DTR';
    public const PROVIDER_DPD = 'DPD_POINT';
    public const PROVIDER_DHL = 'DHL_POINT';
    public const PROVIDER_UPS_ACCESS_POINT = 'UPS_AP_POINT_POINT';
    public const PROVIDER_POCZTEX_PUNKT = 'POCZTEX_PUNKT';


    public function __construct()
    {
        add_filter('flexible_shipping_method_settings', [$this, 'flexibleShippingMethodSettings'], 10, 2);
        add_filter('flexible_shipping_process_admin_options', [$this, 'flexibleShippingProcessAdminOptions']);
    }

    public function flexibleShippingMethodSettings(array $settings, array $shipping_method): array
    {
        $settings['pk_flexible_shipping_points_map_title'] = [
            'title' => 'Mapa punktów odbioru przesyłki Polkurier.pl',
            'type' => 'title',
        ];

        $settings['pk_flexible_shipping_points_map_inpost'] = [
            'label' => 'Pokaż punkty InPost',
            'type' => 'checkbox',
            'description' => 'Wyświetla paczkomaty oraz paczko-punkty InPost na mapie wyboru punktu odbioru przesyłki',
        ];

        $settings['pk_flexible_shipping_points_map_orlen'] = [
            'label' => 'Pokaż punkty Orlen Paczka',
            'type' => 'checkbox',
            'description' => 'Wyświetla punkty Orlen Paczka na mapie wyboru punktu odbioru przesyłki',
        ];

        $settings['pk_flexible_shipping_points_map_dpd'] = [
            'label' => 'Pokaż punkty DPD',
            'type' => 'checkbox',
            'description' => 'Wyświetla punkty DPD na mapie wyboru punktu odbioru przesyłki',
        ];

        $settings['pk_flexible_shipping_points_map_fedex'] = [
            'label' => 'Pokaż punkty FedEx',
            'type' => 'checkbox',
            'description' => 'Wyświetla punkty FedEx na mapie wyboru punktu odbioru przesyłki',
        ];

        $settings['pk_flexible_shipping_points_map_dhl'] = [
            'label' => 'Pokaż punkty DHL',
            'type' => 'checkbox',
            'description' => 'Wyświetla punkty DHL na mapie wyboru punktu odbioru przesyłki',
        ];

        $settings['pk_flexible_shipping_points_map_ups'] = [
            'label' => 'Pokaż punkty UPS',
            'type' => 'checkbox',
            'description' => 'Wyświetla punkty UPS na mapie wyboru punktu odbioru przesyłki',
        ];

        $settings['pk_flexible_shipping_points_map_pocztex'] = [
            'label' => 'Pokaż punkty Pocztex',
            'type' => 'checkbox',
            'description' => 'Wyświetla punkty Pocztex na mapie wyboru punktu odbioru przesyłki',
        ];
        return $settings;
    }

    public function flexibleShippingProcessAdminOptions(array $shipping_method): array
    {
        $request = new Request();
        $shipping_method['pk_flexible_shipping_points_map_inpost'] = $request->request->getInt('woocommerce_flexible_shipping_pk_flexible_shipping_points_map_inpost') === 1 ? 'yes' : 'no';
        $shipping_method['pk_flexible_shipping_points_map_orlen'] = $request->request->getInt('woocommerce_flexible_shipping_pk_flexible_shipping_points_map_orlen') === 1 ? 'yes' : 'no';
        $shipping_method['pk_flexible_shipping_points_map_dpd'] = $request->request->getInt('woocommerce_flexible_shipping_pk_flexible_shipping_points_map_dpd') === 1 ? 'yes' : 'no';
        $shipping_method['pk_flexible_shipping_points_map_fedex'] = $request->request->getInt('woocommerce_flexible_shipping_pk_flexible_shipping_points_map_fedex') === 1 ? 'yes' : 'no';
        $shipping_method['pk_flexible_shipping_points_map_dhl'] = $request->request->getInt('woocommerce_flexible_shipping_pk_flexible_shipping_points_map_dhl') === 1 ? 'yes' : 'no';
        $shipping_method['pk_flexible_shipping_points_map_ups'] = $request->request->getInt('woocommerce_flexible_shipping_pk_flexible_shipping_points_map_ups') === 1 ? 'yes' : 'no';
        $shipping_method['pk_flexible_shipping_points_map_pocztex'] = $request->request->getInt('woocommerce_flexible_shipping_pk_flexible_shipping_points_map_pocztex') === 1 ? 'yes' : 'no';
        return $shipping_method;
    }

    public function onCheckoutProcess(?WC_Order $order = null): void
    {
        $method = Polkurier::instance()->getCurrentShippingMethod();
        if ($method !== null && $this->shouldShowMapButton($method)) {
            if ($order !== null && empty($order->get_billing_phone())) {
                throw new InvalidArgumentException(__('Numer telefonu jest wymagany', 'polkurier'));
            }

            $pointId = null;
            if (!empty($_POST['polkurier_point_id'])) {
                $pointId = $_POST['polkurier_point_id'];
            }
            if (empty($pointId)) {
                $sessionStore = WC()->session->get(Polkurier::NAME . '_delivery_point');
                if (!empty($sessionStore['id'])) {
                    $pointId = $sessionStore['id'];
                }
            }
            if (empty($pointId) && WC()->cart->needs_shipping()) {
                throw new InvalidArgumentException(__('Wybierz punkt odbioru przesyłki', 'polkurier'));
            }
        }
    }

    public function shouldShowMapButton(WC_Shipping_Method $method): bool
    {
        return
            isset($method->instance_settings) &&
            (
                Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_inpost') === 'yes' ||
                Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_orlen') === 'yes' ||
                Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_dpd') === 'yes' ||
                Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_fedex') === 'yes' ||
                Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_dhl') === 'yes' ||
                Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_ups') === 'yes' ||
                Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_pocztex') === 'yes'
            );
    }

    public function getMapProviders(WC_Shipping_Method $method): array
    {
        $providers = [];
        if (isset($method->instance_settings)) {
            if (Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_inpost') === 'yes') {
                $providers[] = self::PROVIDER_INPOST;
            }
            if (Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_orlen') === 'yes') {
                $providers[] = self::PROVIDER_PACZKA_W_RUCHU;
            }
            if (Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_dpd') === 'yes') {
                $providers[] = self::PROVIDER_DPD;
            }
            if (Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_fedex') === 'yes') {
                $providers[] = self::PROVIDER_FEDEX;
            }
            if (Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_dhl') === 'yes') {
                $providers[] = self::PROVIDER_DHL;
            }
            if (Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_ups') === 'yes') {
                $providers[] = self::PROVIDER_UPS_ACCESS_POINT;
            }
            if (Arr::get($method->instance_settings, 'pk_flexible_shipping_points_map_pocztex') === 'yes') {
                $providers[] = self::PROVIDER_POCZTEX_PUNKT;
            }
        }
        return $providers;
    }

    public function getSelectButtonLabel(WC_Shipping_Method $method): string
    {
        $providers = $this->getMapProviders($method);
        if (count($providers) === 1 && $providers[0] === self::PROVIDER_INPOST) {
            return __('Wybierz paczkomat', 'polkurier');
        }
        return __('Wybierz punkt', 'polkurier');
    }

    public function renderShippingMethod(WC_Shipping_Method $method): void
    {
        $tokenManager = new MapsTokenManager();
        $isCod = WC()->session->get('chosen_payment_method') === 'cod';
        $wasSelected = $this->getPostData('polkurier_method_name') === self::ID;

        $selectedProvider = (string)$this->getPostData('polkurier_provider');
        $selectedPointCod = (bool)$this->getPostData('polkurier_point_cod');
        $selectedPointId = (string)$this->getPostData('polkurier_point_id');
        $selectedPointLabel = (string)$this->getPostData('polkurier_point_label');
        $selectedPointType = (bool)$this->getPostData('polkurier_point_type');

        $providers = $this->getMapProviders($method);
        if (!$wasSelected || ($isCod && !$selectedPointCod) || !in_array($selectedProvider, $providers, true)) {
            $selectedPointCod = false;
            $selectedPointId = '';
            $selectedPointLabel = '';
        }

        $position = get_option('polkurier_layout_checkout_point_button_position');
        if (!empty(CheckoutPointButtonPosition::MAP_TEMPLATE[$position])) {
            $template = CheckoutPointButtonPosition::MAP_TEMPLATE[$position];
        } else {
            $template = 'checkout/select_shipping_point.php';
        }

        echo View::render($template, [
            'searchAddress' => '',
            'selectedProvider' => $selectedProvider,
            'selectedPointId' => $selectedPointId,
            'selectedPointLabel' => $selectedPointLabel,
            'selectedPointCod' => $selectedPointCod,
            'selectedPointType' => $selectedPointType,
            'apiToken' => (string)$tokenManager->getToken(),
            'providerName' => $providers,
            'isCod' => $isCod,
            'methodId' => self::ID,
            'buttonLabel' => 'Wybierz punkt dostawy',
        ]);
    }

}
