<?php
namespace Polkurier\Order;

use InvalidArgumentException;
use RuntimeException;
use WC_Order;

class OrderShippingPointManager
{

    /**
     * @throws InvalidArgumentException
     * @throws RuntimeException
     */
    public function save(ShippingPoint $point): ShippingPoint
    {
        if ((int)$point->wp_order_id <= 0) {
            throw new InvalidArgumentException('$point must have a valid order id');
        }

        $order = wc_get_order($point->wp_order_id);
        if ($order) {
            $order->update_meta_data('_polkurier_point_id', $point->code);
            $order->update_meta_data('_polkurier_provider', $point->provider);
            $order->update_meta_data('_polkurier_point_label', $point->label);
            $order->update_meta_data('_polkurier_point_type', $point->type);
            $order->save();
        }

        return $point;
    }

    /**
     * @param WC_Order|int $orderOrId
     */
    public function get($orderOrId): ?ShippingPoint
    {
        if ($orderOrId instanceof WC_Order) {
            $order = $orderOrId;
        } else {
            $order = wc_get_order($orderOrId);
        }

        if (empty($order)) {
            return null;
        }

        $point = new ShippingPoint();
        $point->wp_order_id = $order->get_id();

        // Pobiera z order meta
        if (!empty($order->get_meta('_polkurier_point_id'))) {
            $point->code = (string)$order->get_meta('_polkurier_point_id');
            $point->provider = (string)$order->get_meta('_polkurier_provider');
            $point->label = (string)$order->get_meta('_polkurier_point_label');
            $point->type = (string)$order->get_meta('_polkurier_point_type');
        }

        // Legacy: pobiera z post meta
        if (empty($point->code)) {
            $point->code = (string)get_post_meta($order->get_id(), '_polkurier_point_id', true);
            $point->provider = (string)get_post_meta($order->get_id(), '_polkurier_provider', true);
            $point->label = (string)get_post_meta($order->get_id(), '_polkurier_point_label', true);
            $point->type = (string)get_post_meta($order->get_id(), '_polkurier_point_type', true);
        }

        // Legacy
        if (empty($point->code)) {
            $point->code = (string)get_post_meta($order->get_id(), '_parcel_machine_id', true);
            $point->label = (string)get_post_meta($order->get_id(), '_parcel_machine_desc', true);
        }

        // Dane o punkcie odbioru ze wtyczki "Inpost Paczkomaty" by Damian Ziarnik
        // @see https://pl.wordpress.org/plugins/inpost-paczkomaty/
        if (empty($point->code)) {
            $point->code = (string)get_post_meta($order->get_id(), 'paczkomat_key', true);
            $point->label = (string)get_post_meta($order->get_id(), 'Wybrany paczkomat', true);
            if (!empty($point->code)) {
                $point->provider = 'INPOST_PACZKOMAT';
            }
        }

        if (!empty($point->code)) {
            return $point;
        }

        return null;
    }

}
