<?php

namespace Polkurier;

use DateTime;
use Polkurier\Types\ConfigType;
use Polkurier\Util\Dates;
use Polkurier\Util\ParameterBag;

class PluginInfoProvider
{

    private static ?ParameterBag $manifest = null;

    private const TRANSIENT_KEY = 'PolkurierUpdateManifest';
    private const MANIFEST_URL = '/updatemodule/woocommerce/manifest.json';
    private const UPDATE_INTERVAL = 10800; // 3H

    public function clearCache(): PluginInfoProvider
    {
        delete_transient(self::TRANSIENT_KEY);
        return $this;
    }

    private function getVersionManifest(): ?ParameterBag
    {
        if (self::$manifest === null) {
            $config = new ConfigManager();
            $remote = get_transient(self::TRANSIENT_KEY);
            if ($remote === false) {
                $remote = wp_remote_get($config->getString(ConfigType::POLKURIER_URL) . self::MANIFEST_URL, [
                    'timeout' => 10,
                    'headers' => [
                        'Accept' => 'application/json'
                    ]
                ]);
                if (is_wp_error($remote) || wp_remote_retrieve_response_code($remote) !== 200 || empty(wp_remote_retrieve_body($remote))) {
                    return null;
                }
                set_transient(self::TRANSIENT_KEY, $remote, self::UPDATE_INTERVAL);
            }
            self::$manifest = new ParameterBag((array)json_decode(wp_remote_retrieve_body($remote), true));
        }
        return self::$manifest;
    }

    public function getUpdateInfo(): ?PluginInfo
    {
        $manifest = $this->getVersionManifest();
        if ($manifest === null) {
            return null;
        }
        $pluginInfo = new PluginInfo();
        $pluginInfo->version = (string)$manifest->get('version');
        $pluginInfo->tested = (string)$manifest->get('tested');
        $pluginInfo->date = Dates::dateTimeOrNull((string)$manifest->get('date'));
        if ($pluginInfo->date === null) {
            $pluginInfo->date = new DateTime('1970-01-01 00:00:00');
        }
        $pluginInfo->package_url = (string)$manifest->get('package_url');
        $pluginInfo->readme_url = (string)$manifest->get('readme_url');
        $pluginInfo->icon_url = (string)$manifest->get('icon_url');
        $pluginInfo->changelog = (string)$manifest->get('changelog');
        $pluginInfo->require = (array)$manifest->get('require', []);
        return $pluginInfo;
    }

}
