import {html, LitElement} from "../esm/libs/lit.dist.js?ver=2.4.4";
import styles from '../esm/custom-elements/styles.css.js?ver=2.4.4';
import {objectToOptions} from '../esm/custom-elements/pk-select.js?ver=2.4.4';
import {ProviderController} from "../esm/lit-controllers/provider-controller.js?ver=2.4.4";
import {BankAccountsProvider} from "../esm/providers/bank-accounts-provider.js?ver=2.4.4";
import {formatCurrency, formatIban, numberFilter} from "../esm/util/numbers.js?ver=2.4.4";

const COD_TYPES = {
    'S': 'STANDARD – Zwrot pobrania od 5 do 7 dni roboczych w zależności od przewoźnika',
    '1D': 'Zwrot pobrania w 1 dzień roboczy od daty dostarczenia przesyłki',
    '4D': 'Zwrot pobrania w 4 dni robocze od daty dostarczenia przesyłki',
    '16D': 'Zwrot w 16 dni roboczych od daty dostarczenia przesyłki',
};

const COD_RETURN = {
    BA: 'Przelew na konto bankowe',
    PO: 'Przekaz pocztowy na adres nadawcy',
    MB: 'Do skarbonki',
};

export class PkQuickOrderCodEditView extends LitElement {

    static properties = {
        cod: {type: Object},
    };

    static styles = styles;

    bankAccountsProvider = ProviderController.create(this, new BankAccountsProvider(), {
        load: false,
    });

    setCod(props) {
        this.cod = {
            ...this.cod,
            ...props,
        };
    }

    onClickCancelButton() {
        this.dispatchEvent(new CustomEvent('cancel'));
    }

    onClickAcceptButton() {
        this.dispatchEvent(new CustomEvent('confirm'));
    }

    render() {
        return html`
            <div class="d-flex-column gap">
                <div>
                    <label class="pk-label">
                        <strong>Wartość pobrania</strong>
                        <input 
                            class="pk-input" 
                            placeholder="0,00"
                            .value="${this.cod?.codamount ? formatCurrency(this.cod?.codamount) : ''}"
                            @change="${e => this.setCod({codamount: numberFilter(e.target.value) || null})}"
                        >
                    </label>
                </div>
                <div>
                    <label class="pk-label">
                        <strong>Czas zwrotu pobrania</strong>
                        <pk-select 
                            .value="${this.cod?.codtype}"
                            .options="${objectToOptions(COD_TYPES)}"
                            @select="${e => this.setCod({codtype: e.target.value})}"
                            empty=""
                        ></pk-select>
                    </label>
                </div>
                <div>
                    <label class="pk-label">
                        <strong>Zwrot na</strong>
                        <pk-select
                            .value="${this.cod?.return_cod}"
                            .options="${objectToOptions(COD_RETURN)}"
                            @select="${e => this.setCod({return_cod: e.target.value})}"
                            empty=""
                        ></pk-select>
                    </label>
                </div>
                ${this.cod?.return_cod === 'BA' ? html`
                    <div>
                        <label class="pk-label">
                            <strong>Konto bankowe do zwrotu pobrania</strong>
                            <pk-select
                                .value="${this.cod?.codbankaccount}"
                                .options="${this.bankAccountsProvider.all()}"
                                @select="${e => this.setCod({codbankaccount: e.target.value})}"
                                values="account"
                                .labels="${account => `${account.alias} – ${formatIban(account.account)}`}"
                            ></pk-select>
                        </label>
                    </div>
                    ${this.bankAccountsProvider.length === 0 ? html`
                        <div>
                            <pk-alert type="danger">
                                Brak dostępnych kont bankowych.
                                Przejdź do <a href="admin.php?page=polkurier&tab=bank_accounts" target="_blank">ustawień</a>, aby dodać konto do zwrotu pobrania.
                            </pk-alert>
                        </div>
                    ` : null}
                ` : null}
                <div class="d-flex gap-sm mt" style="justify-content: end;">
                    <button class="btn" @click="${this.onClickCancelButton}">Anuluj</button>
                    <button class="btn btn-primary" @click="${this.onClickAcceptButton}">Zatwierdź</button>
                </div>
            </div>
        `;
    }
}

customElements.define("pk-quick-order-cod-edit-view", PkQuickOrderCodEditView);
