import {html, LitElement} from "../esm/libs/lit.dist.js?ver=2.4.4";
import globalStyles from '../esm/custom-elements/styles.css.js?ver=2.4.4';
import styles from './pk-quick-order-view.css.js?ver=2.4.4';
import {TemplatesProvider} from "../esm/providers/templates-provider.js?ver=2.4.4";
import {ProviderController} from "../esm/lit-controllers/provider-controller.js?ver=2.4.4";
import {CouriersProvider} from "../esm/providers/couriers-provider.js?ver=2.4.4";
import {formatCurrency} from "../esm/util/numbers.js?ver=2.4.4";
import '../esm/custom-elements/pk-badge.js?ver=2.4.4';
import '../esm/custom-elements/pk-icon.js?ver=2.4.4';
import '../esm/custom-elements/pk-toggle.js?ver=2.4.4';

const COD_TYPE_DESCRIPTION = {
    S: 'w 5 do 7 dni roboczych',
    '1D': 'w 1 dzień roboczy',
    '2D': 'w 2 dni robocze',
    '4D': 'w 4 dni robocze',
    '16D': 'w 16 dni roboczych',
};

const COD_RETURN_DESCRIPTION = {
    BA: 'przelewem',
    PO: 'przekazem pocztowym',
    MB: 'do skarbonki',
};

export class PkQuickOrderPackView extends LitElement {

    static styles = [
        globalStyles,
        styles,
    ];

    static properties = {
        order: {},
        shopOrder: {},
        evaluation: {},
        isEvaluated: {type: Boolean},
        isEvaluationRequested: {type: Boolean},
        showResetDefaultCourierButton: {type: Boolean},
    }

    templatesProvider = ProviderController.create(this, new TemplatesProvider());
    couriersProvider = ProviderController.create(this, new CouriersProvider());


    get template() {
        return this.templatesProvider.get(this.order.templateId);
    }

    get courier() {
        return this.couriersProvider.get(this.order.courier);
    }

    setOrder(props) {
        this.order = {
            ...this.order,
            ...props,
        };
        this.dispatchEvent(new CustomEvent('change'));
    }

    hasService(service) {
        return this.order?.courierservice?.[service] === true;
    }

    toggleService(service, toggle) {
        this.setOrder({
            courierservice: {
                ...(this.order?.courierservice || {}),
                [service]: toggle,
            }
        })
    }

    onClickResetDefaultCourierButton() {
        this.dispatchEvent(new CustomEvent('clickResetDefaultCourier'));
    }

    onClickEditButton() {
        this.dispatchEvent(new CustomEvent('clickEditButton'));
    }

    onClickEditCoverAddressButton() {
        this.dispatchEvent(new CustomEvent('clickEditCoverAddressButton'));
    }

    onClickEditCodButton() {
        this.dispatchEvent(new CustomEvent('clickEditCodButton'));
    }

    onClickEditInsuranceButton() {
        this.dispatchEvent(new CustomEvent('clickEditInsuranceButton'));
    }

    onClickEditDescriptionButton() {
        this.dispatchEvent(new CustomEvent('clickEditDescriptionButton'));
    }

    render() {
        const currentTemplate = this.template;
        return html`
            <div class="d-flex gap">
                <div>
                    <pk-icon icon="parcel-taped"></pk-icon>
                </div>
                <div style="flex-grow: 1;">
                    <pk-grid>
                        <pk-col style="--size-lg: 6">
                            <div>
                                <div class="d-flex gap-sm" style="align-items: center">
                                    Przewoźnik:
                                    <pk-badge type="${this.order.courier ? 'warning' : 'danger'}" size="small">
                                        ${!this.order.courier ? html`
                                            <pk-icon type="plain" size="text" icon="exclamation-triangle"></pk-icon>
                                        ` : null}
                                        ${this.couriersProvider.get(this.order.courier)?.name || this.order.courier || 'Nie wybrano przewoźnika'}
                                    </pk-badge>
                                    <button class="btn btn-raised btn-rounded btn-icon" @click="${this.onClickEditButton}" title="Zmień przewoźnika">
                                        <pk-icon icon="pencil" size="text" type="plain" ></pk-icon>
                                    </button>
                                    ${this.showResetDefaultCourierButton ? html`
                                        <button class="btn btn-raised btn-rounded btn-icon" @click="${this.onClickResetDefaultCourierButton}" title="Ustaw domyślny">
                                            <pk-icon icon="undo" size="text" type="plain" ></pk-icon>
                                        </button>
                                    ` : null}
                                </div>
                                <div class="d-flex gap-xs">
                                    <div>
                                        Typ:
                                    </div>
                                    <div>
                                        ${currentTemplate ? html`
                                            <div>
                                                <strong>${currentTemplate.type_description}</strong>
                                                (${currentTemplate.height}×${currentTemplate.width}×${currentTemplate.length}cm, ${currentTemplate.weight}kg)
                                                ${currentTemplate.amount > 1 ? ` × ${currentTemplate.amount}` : null}
                                            </div>
                                        ` : '–'}
                                    </div>
                                </div>
                                <div>
                                    Zawartość: <strong>${this.order.description}</strong>
                                    <button class="btn btn-raised btn-rounded btn-icon" @click="${this.onClickEditDescriptionButton}" title="Zmień opis">
                                        <pk-icon icon="pencil" size="text" type="plain" ></pk-icon>
                                    </button>
                                </div>
                                <div class="d-flex gap-xs">
                                    <div>
                                        Pobranie: 
                                    </div>
                                    <div>
                                        ${this.order?.cod?.codamount > 0 ? html`
                                            <strong>${formatCurrency(this.order.cod.codamount)} zł</strong>
                                        ` : '–'}
                                        <button class="btn btn-raised btn-rounded btn-icon" @click="${this.onClickEditCodButton}" title="Zmień pobranie">
                                            <pk-icon icon="pencil" size="text" type="plain" ></pk-icon>
                                        </button>
                                        ${this.order?.cod?.codamount ? html`
                                            ${this.order?.cod?.return_cod === 'BA' && !this.order?.cod?.codbankaccount ? html`
                                                <div class="text-danger">Brak numeru konta do zwrotu pobrania</div>
                                            ` : null}
                                            <div class="fs-nano">
                                                (Zwrot pobrania ${COD_RETURN_DESCRIPTION[this.order.cod.return_cod]} ${COD_TYPE_DESCRIPTION[this.order.cod.codtype]})
                                            </div>
                                        ` : null}
                                    </div>
                                </div>
                                <div class="d-flex gap-xs align-items-center">
                                    <div>
                                        Ubezpieczenie:
                                    </div>
                                    <div>
                                        <strong>${this.order.insurance ? formatCurrency(this.order.insurance) + ' zł' : '–'}</strong>
                                        <button class="btn btn-raised btn-rounded btn-icon" @click="${this.onClickEditInsuranceButton}" title="Zmień pobranie">
                                            <pk-icon icon="pencil" size="text" type="plain" ></pk-icon>
                                        </button>
                                    </div>
                                </div>
                                <div class="d-flex gap-xs align-items-center">
                                    <div>
                                        Cena przesyłki: 
                                    </div>
                                    ${this.isEvaluationRequested ? html`
                                        <pk-spinner size="xs"></pk-spinner>
                                    ` : null}
                                    <div>
                                        ${this.evaluation ? html`
                                            <span class="text-nowrap">
                                                <strong>${formatCurrency(this.evaluation.netprice)}</strong> zł netto
                                            </span>
                                            <span class="text-nowrap">
                                                (<strong>${formatCurrency(this.evaluation.grossprice)}</strong> zł brutto)
                                            </span>
                                        ` : null}
                                        ${!this.evaluation && !this.isEvaluationRequested ? html`–` : null}
                                    </div>
                                </div>
                                ${this.isEvaluated && !this.evaluation && !this.isEvaluationRequested && this.order.courier ? html`
                                    <div class="d-inline-flex gap-sm text-danger">
                                        <pk-icon type="plain" size="text" icon="exclamation-triangle"></pk-icon>
                                        <div>Nie można wysłać wybranym przewoźnikiem</div>
                                    </div>
                                ` : null}
                            </div>
                        </pk-col>
                        <pk-col style="--size-lg: 6">
                            <div class="d-flex-column fs-nano">
                                <pk-toggle
                                    class="pk-toggle-small"
                                    .checked="${this.order.insurance > 0}"
                                    .disabled="${this.shopOrder?.total_value <= 0}"
                                    @change="${e => {
                                        this.setOrder({
                                            insurance: e.target.checked 
                                                ? Math.max(0, this.shopOrder?.total_value, this.order?.cod?.codamount)
                                                : 0,
                                        });
                                    }}"
                                >
                                    Ubezpieczenie
                                </pk-toggle>
                                <pk-toggle 
                                    class="pk-toggle-small"
                                    .checked="${this.hasService('ROD')}"
                                    @change="${e => this.toggleService('ROD', e.target.checked)}"
                                >
                                    Dokumenty zwrotne
                                </pk-toggle>
                                <pk-toggle
                                    class="pk-toggle-small"
                                    .checked="${this.hasService('WEEK_COLLECTION')}"
                                    @change="${e => this.toggleService('WEEK_COLLECTION', e.target.checked)}"
                                >
                                    Dostawa weekendowa (InPost)
                                </pk-toggle>
                                <pk-toggle
                                    class="pk-toggle-small"
                                    .checked="${this.hasService('COURIER_WITH_LABEL')}"
                                    @change="${e => this.toggleService('COURIER_WITH_LABEL', e.target.checked)}"
                                >
                                    Podjazd kuriera z wydrukowaną etykietą (DHL)
                                </pk-toggle>
                                <pk-toggle
                                    class="pk-toggle-small"
                                    .checked="${this.hasService('SMS_NOTIFICATION_RECIPIENT')}"
                                    @change="${e => {
                                        this.toggleService('SMS_NOTIFICATION_RECIPIENT', e.target.checked)
                                        this.toggleService('SMS_NOTIFICATION_RECIPIENT_WITH_NAME', false)
                                    }}"
                                >
                                    Powiadomienie SMS dla odbiorcy
                                </pk-toggle>
                                <pk-toggle
                                    class="pk-toggle-small"
                                    .checked="${this.hasService('SMS_NOTIFICATION_RECIPIENT_WITH_NAME')}"
                                    @change="${e => {
                                        this.toggleService('SMS_NOTIFICATION_RECIPIENT', false)
                                        this.toggleService('SMS_NOTIFICATION_RECIPIENT_WITH_NAME', e.target.checked)
                                    }}"
                                >
                                    Powiadomienie SMS dla odbiorcy z własną nazwą
                                </pk-toggle>
                                <pk-toggle
                                    class="pk-toggle-small"
                                    .checked="${this.hasService('COVER_ADDRESS_SENDER')}"
                                    @change="${e => this.toggleService('COVER_ADDRESS_SENDER', e.target.checked)}"
                                >
                                    Inny adres nadawcy na etykiecie

                                    ${this.hasService('COVER_ADDRESS_SENDER') ? html`
                                        <div slot="description">
                                            ${this.order?.cover_address ? html`
                                                <div>
                                                    <div class="fw-bold d-flex gap-sm">
                                                        ${this.order?.cover_address?.company || ''}
                                                        ${this.order?.cover_address?.person || ''}
                                                        <button class="btn btn-raised btn-rounded btn-icon" @click="${this.onClickEditCoverAddressButton}" title="Zmień adres">
                                                            <pk-icon icon="pencil" size="text" type="plain" ></pk-icon>
                                                        </button>
                                                    </div>
                                                    <div>
                                                        ${this.order?.cover_address?.street || ''} 
                                                        ${this.order?.cover_address?.housenumber || ''}
                                                        ${this.order?.cover_address?.flatnumber ? ` / ${this.order?.cover_address?.flatnumber}` : ''},
                                                        ${this.order?.cover_address?.postcode || ''} ${this.order?.cover_address?.city || ''} (${this.order?.country || 'PL'})
                                                    </div>
                                                    <div>${this.order?.cover_address?.phone || ''}, ${this.order?.cover_address?.email || ''}</div>
                                                </div>
                                            ` : html`
                                                <span class="text-danger">Brak adresu</span>
                                                <button class="btn btn-raised btn-rounded btn-icon" @click="${this.onClickEditCoverAddressButton}" title="Zmień adres">
                                                    <pk-icon icon="plus" size="text" type="plain" ></pk-icon>
                                                </button>
                                            `}
                                        </div>
                                    ` : null}
                                </pk-toggle>
                            </div>
                        </pk-col>
                    </pk-grid>
                </div>
            </div>
        `;
    }
}

customElements.define('pk-quick-order-pack-view', PkQuickOrderPackView);
