import {html, map, LitElement} from "../esm/libs/lit.dist.js?ver=2.4.4";
import styles from "../esm/custom-elements/styles.css.js?ver=2.4.4";
import {ProviderController} from "../esm/lit-controllers/provider-controller.js?ver=2.4.4";
import {TemplatesProvider} from "../esm/providers/templates-provider.js?ver=2.4.4";
import {CouriersProvider} from "../esm/providers/couriers-provider.js?ver=2.4.4";
import '../esm/custom-elements/pk-grid.js?ver=2.4.4';
import '../esm/custom-elements/pk-badge.js?ver=2.4.4';
import '../esm/custom-elements/pk-list.js?ver=2.4.4';
import '../esm/custom-elements/pk-icon.js?ver=2.4.4';
import '../esm/custom-elements/pk-banner.js?ver=2.4.4';

export class PkQuickOrderTemplates extends LitElement {

    static styles = styles;

    static properties = {
        value: {type: String},
        isLoading: {type: Boolean, state: true},
        templates: {},
    }

    templatesProvider = ProviderController.create(this, new TemplatesProvider());
    couriersProvider = ProviderController.create(this, new CouriersProvider());

    get template() {
        return this.templatesProvider.get(this.value);
    }

    getTemplateIcon(template) {
        if (template.shipment_type === 'palette') {
            return 'pallet-stacked';
        }
        if (template.shipment_type === 'envelope') {
            return 'envelope';
        }
        return 'parcel-taped';
    }

    onSelect(template) {
        this.value = template.id;
        this.dispatchEvent(new CustomEvent('change', {
            detail: template,
            bubbles: true,
            composed: true,
        }));
    }

    get hasDefaultTemplate() {
        return this.templatesProvider.find(t => t.default);
    }

    render() {
        if (this.templatesProvider.isRequested() && this.templatesProvider.length === 0) {
            return html`
                <div class="text-center">
                    <pk-spinner size="sm"></pk-spinner>
                </div>
            `;
        }

        if (this.templatesProvider.length === 0) {
            return html`
                <div class="text-center">
                    <pk-icon icon="parcel-problem" style="--pk-icon-size: 8rem"></pk-icon>
                    <p>
                        Nie masz zdefiniowanych szablonów wysyłek.<br>
                        Przejdź do <a href="admin.php?page=polkurier&tab=templates" target="_blank">ustawień</a>, aby dodać nowy szablon.
                    </p>
                </div>
            `;
        }

        return html`
            <pk-list>
                ${map(this.templatesProvider.all(), template => html`
                    <pk-list-item
                        class="cursor-pointer ${template.id === this.value ? 'active' : ''}"
                        @click="${() => this.onSelect(template)}"
                        style="display: flex; gap: 1rem"
                    >
                        <div>
                            <pk-icon icon="${this.getTemplateIcon(template)}" size="sm"></pk-icon>
                        </div>
                        <div>
                            <div>
                                <strong>
                                    ${template.name || html`<span class="text-muted">(Bez nazwy)</span>`}
                                </strong>
                            </div>
                            <div class="text-muted fs-nano">
                                <div>
                                    ${template.type_description}
                                    (${template.height}×${template.width}×${template.length}cm, ${template.weight}kg)
                                </div>
                                ${template.courier ? html`
                                    <pk-badge size="small">
                                        ${this.couriersProvider.get(template.courier)?.name || template.courier}
                                    </pk-badge>
                                ` : null}
                            </div>
                        </div>
                    </pk-list-item>
                `)}
            </pk-list>
            ${!this.hasDefaultTemplate ? html`
                <pk-banner type="info" icon="info-circle" class="mt">
                    Brak domyślnego szablonu. Przejdź do <a href="admin.php?page=polkurier&tab=templates" target="_blank">ustawień</a>,
                    aby wybrać domyślny szablon.
                </pk-banner>
            ` : null}
        `;
    }
}

customElements.define('pk-quick-order-templates', PkQuickOrderTemplates);
