import {html, LitElement, ref, createRef} from "../libs/lit.dist.js?ver=2.4.4";
import {debounce, getViewportSize} from "../util/browser.js?ver=2.4.4";
import styles from './pk-dialog.css.js?ver=2.4.4';

export const MODAL_SIZE_SMALL = 'sm';
export const MODAL_SIZE_MEDIUM = 'md';
export const MODAL_SIZE_LARGE = 'lg';
export const MODAL_SIZE_EXTRA_LARGE = 'xl';
export const MODAL_SIZE_DEFAULT = MODAL_SIZE_LARGE;

export const MODAL_POSITION_DEFAULT = '';
export const MODAL_POSITION_TOP = 'top';
export const MODAL_POSITION_RIGHT = 'right';
export const MODAL_POSITION_BOTTOM = 'bottom';
export const MODAL_POSITION_LEFT = 'left';

export const MODAL_HEADER_STYLE_DEFAULT = 'default';
export const MODAL_HEADER_STYLE_FANCY = 'fancy';

export const MODAL_HEADER_ALIGN_LEFT = 'left';
export const MODAL_HEADER_ALIGN_CENTER = 'center';

export const CLOSE_BUTTON_DEFAULT = 'default';
export const CLOSE_BUTTON_BACK = 'back';
export const CLOSE_BUTTON_NONE = 'none';

export const MODAL_BACKDROP_CLOSE = true;
export const MODAL_BACKDROP_NONE = false;
export const MODAL_BACKDROP_STATIC = 'static';
export const MODAL_BACKDROP_DEFAULT = MODAL_BACKDROP_CLOSE;

let nextDialogId = 1;

export const getParentDialog = (el) => {
    return el.closest('pk-dialog');
}

export const closeParentDialog = (el) => {
    const dialog = getParentDialog(el);
    dialog?.close();
    return dialog;
}

const handleBodyClass = debounce(() => {
    let hasOpenedModals = false;
    for (let modal of PkDialog.dialogs) {
        if (modal.opened) {
            hasOpenedModals = true;
            break;
        }
    }
    document.body.classList.toggle('pk-body--has-opened-pk-modals', hasOpenedModals);
}, 50)

export class PkDialog extends LitElement {

    static minDragAmount = 50;
    static dragScreenPadding = 25;

    static dialogs = [];

    #clickStartedOnDialog = false;
    innerDialogRef = createRef();
    dialogRef = createRef();
    bodyRef = createRef();

    static styles = styles;

    static get properties() {
        return {
            opened: {type: Boolean},
            title: {type: String},
            header: {type: Boolean},
            headerStyle: {type: String},
            headerAlign: {type: String},
            size: {type: String},
            position: {type: String},
            closeButton: {type: String},
            minimizeButton: {type: Boolean},
            backdrop: {type: String},
            draggable: {type: Boolean},
            isVisible: {type: Boolean, state: true},
            isContentVisible: {type: Boolean, state: true},
        };
    }

    get dialog() {
        return this.dialogRef.value;
    }

    get innerDialog() {
        return this.innerDialogRef.value;
    }

    get body() {
        return this.bodyRef.value;
    }

    static create(properties = {}) {
        const element = document.createElement('pk-dialog');
        for (let propertyName in properties) {
            element[propertyName] = properties[propertyName];
        }
        return element;
    }

    constructor() {
        super();
        this.cid = 'pk-dialog-' + (nextDialogId++);
        this.title = '';
        this.content = '';
        this.header = true;
        this.headerStyle = MODAL_HEADER_STYLE_DEFAULT;
        this.headerAlign = MODAL_HEADER_ALIGN_LEFT;
        this.size = MODAL_SIZE_DEFAULT;
        this.position = MODAL_POSITION_DEFAULT;
        this.closeButton = CLOSE_BUTTON_DEFAULT;
        this.minimizeButton = false;
        this.backdrop = MODAL_BACKDROP_DEFAULT;
        this.draggable = true;
        this.isOpened = false;
        this.createdAt = new Date();
    }

    open() {
        this.opened = true;
    }

    close() {
        this.opened = false;
    }

    dispatchViewEvent(eventName, detail = {}, options = {}) {
        const event = new CustomEvent(eventName, {
            cancelable: true,
            bubbles: true,
            composed: true,
            ...options,
            detail,
        });
        this.dispatchEvent(event);
        return event;
    }

    updated(changedProps) {
        super.updated(changedProps);

        if (changedProps.has('opened')) {
            if (this.opened) {
                this.isContentVisible = true;
                setTimeout(() => this.classList.add('opened'));
                this.dispatchViewEvent('open');
                setTimeout(() => this.isVisible = true, 150);
            } else {
                const e = this.dispatchViewEvent('close');
                if (e.defaultPrevented) {
                    this.opened = true;
                } else {
                    this.classList.remove('opened');
                    setTimeout(() => this.isVisible = false, 175);
                }
            }
        }

        if (changedProps.has('isVisible')) {
            if (this.isVisible) {
                this.dispatchViewEvent('opened');
            } else {
                this.isContentVisible = false;
                this.classList.remove('opened');
                this.dispatchViewEvent('closed');
            }
        }
    }

    connectedCallback() {
        super.connectedCallback();
        this.constructor.dialogs.push(this);
        handleBodyClass();
        this.addEventListener('open', handleBodyClass);
        this.addEventListener('opened', handleBodyClass);
        this.addEventListener('close', handleBodyClass);
        this.addEventListener('closed', handleBodyClass);
    }

    disconnectedCallback() {
        super.disconnectedCallback();
        this.constructor.dialogs.splice(this.constructor.dialogs.indexOf(this), 1);
        handleBodyClass();
    }

    handleDragStart(e) {
        if (this.draggable && this.innerDialog && this.position === MODAL_POSITION_DEFAULT && e.buttons === 1) {
            const rect = this.innerDialog.getBoundingClientRect();
            this.dragging = {
                startX: rect.x,
                startY: rect.y,
                offsetX: e.clientX - rect.x,
                offsetY: e.clientY - rect.y,
            };
        }
    }

    handleDragStop() {
        this.dragging = null;
    }

    handleDragging(e) {
        if (this.dragging && this.innerDialog && !this.__updatePositionRequested) {
            this.__updatePositionRequested = true;
            window.requestAnimationFrame(() => {
                this.__updatePositionRequested = false;
                if (!this.dragging) {
                    return;
                }
                const viewport = getViewportSize();
                const rect = this.innerDialog.getBoundingClientRect();

                const minX = 0 - rect.width + PkDialog.dragScreenPadding;
                const maxX = viewport.width - PkDialog.dragScreenPadding;
                const maxY = viewport.height - PkDialog.dragScreenPadding;
                const nextX = Math.min(maxX, Math.max(minX, e.clientX - this.dragging.offsetX));
                const nextY = Math.min(maxY, Math.max(0, e.clientY - this.dragging.offsetY));

                if (Math.abs(nextX - this.dragging.startX) < PkDialog.minDragAmount && Math.abs(nextY - this.dragging.startY) < PkDialog.minDragAmount) {
                    return;
                }

                this.innerDialog.style.margin = '0';
                this.innerDialog.style.left = nextX + 'px';
                this.innerDialog.style.top = nextY + 'px';
            });
        }
    }

    handleClickBackdrop(e) {
        if (!this.isVisible || e.x === 0 || e.y === 0) {
            return;
        }

        const els = this.renderRoot.elementsFromPoint(e.x, e.y);
        for (let el of els) {
            if (el.classList.contains('modal-dialog') || el.classList.contains('modal-content')) {
                return;
            }
        }

        if (this.isVisible && !this.#clickStartedOnDialog) {
            this.opened = false
        }
    }

    handleKeyDown(e) {
        if (this.isVisible && e.key === 'Escape') {
            this.opened = false;
        }
    }

    handleScroll(e) {
        this.dispatchViewEvent('scroll', e);
    }

    #handleMouseDownOnDialog(e) {
        e.stopImmediatePropagation();
        this.#clickStartedOnDialog = true;
    }

    #handleMouseDownOnBackdrop(e) {
        e.stopImmediatePropagation();
        this.#clickStartedOnDialog = false;
    }

    render() {
        if (!this.isContentVisible) {
            return;
        }

        return html`
            <div
                class="modal"
                tabindex="-1"
                ${ref(this.dialogRef)}
                @click="${this.handleClickBackdrop}"
                @keydown="${this.handleKeyDown}"
                @mousedown="${this.#handleMouseDownOnBackdrop}"
                @mouseup="${this.handleDragStop}"
                @mousemove="${this.handleDragging}"
                @scroll="${this.handleScroll}"
            >
                <div 
                    ${ref(this.innerDialogRef)}
                    @mousedown="${this.#handleMouseDownOnDialog}"
                    class="modal-dialog modal-${this.size} modal-dialog-${this.position || 'default'} ${this.header !== false ? 'modal-with-header' : ''}"
                >
                    <div class="modal-content" ${ref(this.bodyRef)}>
                        <slot name="header">
                            ${this.header !== false ? html`
                                <div class="dialog-header dialog-header-${this.headerStyle}" @mousedown="${this.handleDragStart}">
                                    <slot name="title">
                                        <h5 class="dialog-title">
                                            ${this.title}
                                        </h5>
                                    </slot>
                                    <div class="dialog-controls">
                                        <slot name="buttons"></slot>
                                        <slot name="controls">
                                            ${this.closeButton === CLOSE_BUTTON_DEFAULT ? html`
                                                <button type="button" @click="${() => this.opened = false}">
                                                    &times
                                                </button>
                                            ` : null}
                                        </slot>
                                    </div>
                                </div>
                            ` : null}
                        </slot>
                        <div class="modal-body">
                            <slot></slot>
                        </div>
                        <slot name="footer"></slot>
                    </div>
                </div>
            </div>
        `;
    }
}

customElements.define('pk-dialog', PkDialog);
