import {css, html, unsafeSVG, LitElement} from '../libs/lit.dist.js?ver=2.4.4';
import styles from './styles.css.js?ver=2.4.4';
import './pk-spinner.js?ver=2.4.4';

const ICONS = {};

export class PkIcon extends LitElement {
    static styles = [
        styles,
        css`
            :host {
                --__pk-icon-bg: var(--pk-icon-bg, var(--pk-bg-blue-100));
                --__pk-icon-color: var(--pk-icon-color, var(--pk-blue-700));
                --__pk-icon-size: var(--pk-icon-size, 3rem);
                
                position: relative;
                display: inline-block;
                width: var(--__pk-icon-size);
                height: var(--__pk-icon-size);
                color: var(--__pk-icon-color);
                
                &::before {
                    content: '';
                    position: absolute;
                    top: 0;
                    left: 0;
                    border-radius: 50%;
                    background-color: var(--__pk-icon-bg);
                    width: calc(var(--__pk-icon-size) * 0.75);
                    height: calc(var(--__pk-icon-size) * 0.75);
                    z-index: 5;
                }
            }
            
            
            svg {
                display: inline-block;
                width: 100%;
                height: 100%;
                position: relative;
                z-index: 10;
            }
            
            :host([type="plain"]) {
                --pk-icon-bg: none;
                --pk-icon-color: currentColor;
            }
            
            :host([size="text"]) {
                --pk-icon-size: 2ch;
            }

            :host([size="xs"]) {
                --pk-icon-size: 1rem;
            }
            
            :host([size="sm"]) {
                --pk-icon-size: 2rem;
            }
            
            :host([size="md"]) {
                --pk-icon-size: 2.5rem;
            }
            
            :host([size="lg"]) {
                --pk-icon-size: 4rem;
            }
            
            :host([size="xl"]) {
                --pk-icon-size: 5rem;
            }
            
            :host([size="auto"]) {
                --pk-icon-size: 100%;
            }
        `,
    ];

    static properties = {
        icon: {},
    };

    update(t) {
        super.update(t);
        if (t.has('icon')) {
            this.loadSvg().then();
        }
    }

    async loadSvg() {
        if (!this.icon) {
            return;
        }
        if (ICONS[this.icon]) {
            if (ICONS[this.icon] instanceof Promise) {
                ICONS[this.icon].finally(() => this.requestUpdate());
            }
            return;
        }
        const promise = ICONS[this.icon] = new Promise(async (resolve) => {
            const response = await fetch(`${POLKURIER.assetsUrl}/images/icons/${this.icon}.svg`);
            const contentType = response.headers.get('content-type');

            if (contentType === 'image/svg+xml' || contentType === 'image/svg') {
                ICONS[this.icon] = await response.text();
            } else {
                ICONS[this.icon] = '<svg></svg>'
            }
            resolve();
        });
        promise.finally(() => this.requestUpdate());
    }

    get iconSVG() {
        if (typeof ICONS[this.icon] === 'string') {
            let content = ICONS[this.icon] || '';
            if (content) {
                content = content.replace(/#(0{3,8})/g, 'currentcolor');
            }
            return unsafeSVG(content);
        }
    }

    render() {
        return html`<slot>${this.iconSVG}</slot>`
    }
}

window.customElements.define('pk-icon', PkIcon);
