import {css, LitElement} from "../libs/lit.dist.js?ver=2.4.4";

/**
 * @typedef {'small' | 'large'} PkSpinnerSize
 * @typedef {'primary' | 'secondary' | 'success' | 'danger' | 'warning' | 'info' | 'light' | 'dark'} PkSpinnerColor
 */

/**
 * @tag pk-spinner
 * @property {PkSpinnerSize} size
 * @property {PkSpinnerColor} color
 */
export class PkSpinner extends LitElement {

    static styles = css`
        :host {
            box-sizing: border-box;
            display: inline-block;
            border-radius: 100%;
            height: var(--size, 2rem);
            width: var(--size, 2rem);
            vertical-align: text-bottom;
            border: var(--width, 0.25rem) var(--color, currentColor) solid;
            border-left-color: transparent;
            animation: circle-spinner-animation var(--speed, 0.75s) linear infinite;
        }

        :host([xsmall]),
        :host([size="xs"]),
        :host([size="xsmall"]) {
            --size: 0.6rem;
            --width: 0.15rem;
        }
        
        :host([small]),
        :host([size="sm"]),
        :host([size="small"]) {
            --size: 1rem;
            --width: 0.15rem;
        }

        :host([size="lg"]),
        :host([size="large"]) {
            --size: 3rem;
        }

        :host([center]) { // Legacy
            display: block;
            margin-left: auto;
            margin-right: auto;
        }

        :host([color="primary"]) {
            --color: var(--bs-primary, var(--theme-primary, var(--primary)));
        }

        :host([color="secondary"]) {
            --color: var(--bs-secondary, var(--theme-secondary, var(--secondary)));
        }

        :host([color="success"]) {
            --color: var(--bs-success, var(--theme-success, var(--success)));
        }

        :host([color="danger"]) {
            --color: var(--bs-danger, var(--theme-danger, var(--danger)));
        }

        :host([color="warning"]) {
            --color: var(--bs-warning, var(--theme-warning, var(--warning)));
        }

        :host([color="info"]) {
            --color: var(--bs-info, var(--theme-info, var(--info)));
        }

        :host([color="light"]) {
            --color: var(--bs-light, var(--theme-light, var(--light)));
        }

        :host([color="dark"]) {
            --color: var(--bs-dark, var(--theme-dark, var(--dark)));
        }

        @keyframes circle-spinner-animation {
            100% {
                rotate: 360deg;
            }
        }
    `;

    static properties = {
        size: {type: String, reflect: true},
        color: {type: String, reflect: true},
    }

    render() {
        return null;
    }
}

if (!customElements.get('pk-spinner')) {
    customElements.define('pk-spinner', PkSpinner);
}

function toggleIconsVisibility(element, toggle) {
    const icons = element.querySelectorAll('.fa:first-child, .fas:first-child, .fad:first-child, .far:first-child, .fal:first-child, .fab:first-child, .bi:first-child, .spinner-icon, pk-icon:first-child, fa-icon:first-child');
    for (let icon of icons) {
        if (toggle) {
            icon.style.display = 'inline-block';
        } else {
            icon.style.display = 'none';
        }
    }
}

function normalizeElement(element) {
    if (typeof element === 'string') {
        return document.querySelectorAll(element);
    }
    return element;
}

/**
 * @typedef {object} SetButtonBusyStateOptions
 * @property {PkSpinnerSize} size
 * @property {PkSpinnerColor} color
 * @property {boolean} replace
 */

/**
 * Ustawia stan przycisku jako ładowanie
 * @param {HTMLElement|string} element
 * @param {Partial<SetButtonBusyStateOptions>} options
 */
export const setButtonBusyState = (element, options = {}) => {
    if (!element) {
        return;
    }
    element = normalizeElement(element);
    if (element.length) {
        for (let el of element) {
            setButtonBusyState(el, options);
        }
        return;
    }

    let spinner = element.querySelector('pk-spinner');
    if (!spinner) {
        spinner = document.createElement('pk-spinner');
        spinner.size = options.size || 'small';
        spinner.color = options.color || '';
    }

    let container = element.querySelector('.spinner-icon-container');
    if (!container) {
        container = element;
    }

    element.disabled = true;
    element.setAttribute('disabled', '');

    if (options.replace) {
        container.replaceChildren();
    } else {
        toggleIconsVisibility(element, false);
    }
    container.prepend(spinner);
};

/**
 * Usuwa stan przycisku jako ładowanie
 * @param {HTMLElement|string} element
 */
export const unsetButtonBusyState = element => {
    element = normalizeElement(element);
    if (element.length) {
        for (let el of element) {
            unsetButtonBusyState(el);
        }
        return;
    }

    const spinner = element.querySelector('pk-spinner');
    spinner?.parentElement?.removeChild(spinner);

    element.disabled = false;
    element.removeAttribute('disabled');
    toggleIconsVisibility(element, true);
};

/**
 * Usuwa stan przycisku jako ładowanie
 * @param {HTMLElement|string} element
 * @param {boolean} toggle
 * @param {Partial<SetButtonBusyStateOptions>} options
 */
export const toggleButtonBusyState = (element, toggle, options = {}) => {
    if (toggle === true) {
        return setButtonBusyState(element, options)
    }
    unsetButtonBusyState(element);
};
