import {css, html, LitElement} from '../libs/lit.dist.js?ver=2.4.4';
import './pk-spinner.js?ver=2.4.4';
import './pk-icon.js?ver=2.4.4';

export const DEFAULT_DURATION = 6000;
export const TOAST_TYPE_SUCCESS = 'SUCCESS';
export const TOAST_TYPE_WARNING = 'WARNING';
export const TOAST_TYPE_DANGER = 'DANGER';
export const TOAST_TYPE_INFO = 'INFO';

export class PkToastMessage extends LitElement {

    static styles = css`
        :host {
            margin: 0 auto;
        }

        .toast--danger {
            --color: rgb(145, 0, 0);
            --background: rgba(255, 207, 207, 0.8);
            --button-background: rgba(220, 60, 60, 0.15);
            --button-background-hover: rgba(220, 60, 60, 0.25);
        }

        .toast {
            &.fade-out {
                .toast-body {
                    transition: all 0.3s;
                    opacity: 0;
                    scale: 0.9;
                    @starting-style {
                        scale: 1;
                    }
                }
            }
        }

        .toast-body {
            pointer-events: all;
            display: flex;
            gap: 0.5rem;
            align-items: center;
            width: auto;
            min-width: 5rem;
            margin: 0 auto;
            padding: 0.75rem 2rem;
            backdrop-filter: blur(0.5rem);
            border-radius: 2rem;
            box-shadow: 0 2px 1px rgba(0, 0, 0, 0.33);
            background: var(--background, rgb(70 70 70 / 0.8));
            color: var(--color, white);
            --link-color: var(--color, white);
            font-weight: 400;
            translate: 0 0;
            transition: all 0.5s;
            opacity: 1;
            @starting-style {
                opacity: 0;
                translate: 0 100%;
            }
        }

        .close-button {
            font-weight: 500;
            border: 0 none;
            background: none;
            color: var(--color, white);
            border-radius: 50%;
            width: 2rem;
            height: 2rem;
            line-height: 2rem;
            padding: 0;
            margin: -0.5rem -1.5rem -0.5rem auto;
            flex-shrink: 0;
            transition: background 0.4s;
            &:hover {
                transition: background 0.4s;
                background: var(--button-background, rgb(255 255 255 / 0.25));
            }
        }
        
        slot[name="button"]::slotted(button) {
            font-weight: 400;
            margin: -0.5rem 0 -0.5rem 1rem;
            padding: 0.5rem 0.75rem;
            color: var(--color, white);
            border: 0 none;
            border-radius: 0.4rem;
            background: var(--button-background, rgb(255 255 255 / 0.15));
            transition: background 0.4s;
        }

        slot[name="button"]::slotted(button:hover) {
            transition: background 0.4s;
            background: var(--button-background-hover, rgb(255 255 255 / 0.25));
        }

        .icon {
            align-self: center;
            font-size: 1rem;
            width: 1.2rem;
            height: 1.2rem;
            margin-left: -0.5rem;
            margin-right: 0.25rem;

            &[icon=""] {
                display: none !important;
            }
        }

        pk-spinner {
            --size: 1.2rem;
            --width: 0.2rem;
            align-self: center;
            margin-left: -0.5rem;
            margin-right: 0.25rem;
        }
    `;

    static properties = {
        type: {type: String},
        duration: {type: Number},
        icon: {type: String},
        iconType: {type: String},
        iconCollection: {type: String},
        open: {type: Boolean},
        closable: {type: Boolean},
        spinner: {type: Boolean},
    };

    #timeoutTimer;
    #timeoutTimeElapsed = 0;
    #timeoutTimerStartedAt = 0;

    constructor() {
        super();
        this.open = true;
    }

    connectedCallback() {
        super.connectedCallback();
        this.addEventListener('mouseenter', this.#stopHideTimeout);
        this.addEventListener('mouseleave', this.#startHideTimeout);
    }

    #onTransitionEnd(event) {
        if (event.propertyName === 'opacity' && !this.open) {
            this.dispatchEvent(new Event('hidden'));
        }
        if (event.propertyName === 'opacity' && this.open) {
            this.#startHideTimeout();
        }
    }

    #stopHideTimeout() {
        clearTimeout(this.#timeoutTimer);
        if (this.#timeoutTimerStartedAt > 0) {
            this.#timeoutTimeElapsed += (Date.now() - this.#timeoutTimerStartedAt);
        }
        this.#timeoutTimerStartedAt = 0;
    }

    #startHideTimeout() {
        this.#stopHideTimeout();
        if (this.duration < Infinity) {
            this.#timeoutTimerStartedAt = Date.now();
            this.#timeoutTimer = setTimeout(() => {
                this.#hide();
            }, Math.max(0, this.duration - this.#timeoutTimeElapsed));
        }
    }

    #hide() {
        this.open = false;
    }

    updated(t) {
        super.updated(t);
        if (t.has('open')) {
            if (!this.open) {
                this.dispatchEvent(new Event('hide'));
            }
        }
    }

    get #iconName() {
        if (this.icon) {
            return this.icon;
        }
        switch (this.type) {
            case TOAST_TYPE_SUCCESS:
                return 'check-circle';
            case TOAST_TYPE_WARNING:
                return 'exclamation-triangle';
            case TOAST_TYPE_DANGER:
                return 'exclamation-octagon';
        }
        return '';
    }

    render() {
        return html`
            <div class="toast toast--${this.type.toLowerCase()} ${!this.open ? ' fade-out' : ''}">
                <div
                    class="toast-body"
                    @transitionend="${this.#onTransitionEnd}"
                    @transitioncancel="${this.#onTransitionEnd}"
                >
                    <slot name="icon">
                        ${this.spinner ? html`
                            <pk-spinner></pk-spinner>
                        ` : html`
                            <pk-icon
                                class="icon"
                                icon="${this.#iconName}"
                                type="${this.iconType || 'plain'}"
                            ></pk-icon>
                        `}
                    </slot>
                    <slot></slot>
                    <slot name="button"></slot>
                    ${this.closable !== false ? html`
                        <button @click="${this.#hide}" class="close-button">
                            &times;
                        </button>
                    ` : null}
                </div>
                <slot name="external-button"></slot>
            </div>
        `
    }
}

window.customElements.define('pk-toast-message', PkToastMessage);


export class PkToastContainer extends LitElement {

    static styles = css`
        :host {
            position: fixed;
            bottom: 0;
            left: 50%;
            translate: -50% 0;
            width: 70%;
            padding: 0 0 1.5rem 0;
            display: flex;
            flex-direction: column-reverse;
            gap: 0.3rem;
            pointer-events: none;
            z-index: 99999991;
        }

        @media (max-width: 801px) {
            :host {
                width: 100vw;
                left: 0;
                translate: 0;
                padding-left: 0.5rem;
                padding-right: 0.5rem;
            }
        }
    `;

    render() {
        return html`<slot></slot>`;
    }
}

window.customElements.define('pk-toast-container', PkToastContainer);


let toastsContainer;
const getContainer = () => {
    if (!toastsContainer) {
        toastsContainer = document.getElementById('pk-toast-container');
        if (!toastsContainer) {
            toastsContainer = document.createElement('pk-toast-container');
            toastsContainer.id = 'pk-toast-container';
            document.body.appendChild(toastsContainer);
        }
    }
    return toastsContainer;
};

/**
 * @typedef {object} ToastMessage
 * @property {string} message Treść wiadomości
 * @property {'INFO' | 'DANGER' | 'WARNING' | 'SUCCESS'} type Typ powiadomienia
 * @property {number | undefined} duration Czas trwania powiadomienia, Infinity – nie ukrywa automatycznie
 * @property {string | undefined} actionTitle Etykieta dodatkowego przycisku
 * @property {Function | undefined} actionHandler Akcja dodatkowego przycisku. Automatycznie zamyka powiadomienia.
 * W argumencie zostanie przekazany obiekt Event kliknięcia. Aby pominąć automatyczne zamknięcie można anulować na nim
 * domyślną akcję: `event.preventDefault()`
 * @property {string | undefined} icon Ikona powiadomienia ze zbioru zdefiniowanego w parametrze `collection`
 * @property {'solid' | 'regular' | 'light' | 'dual_tone' | undefined} iconType Typ ikona powiadomienia ze zbioru FontAwesome
 * @property {'fa' | 'bi' | 'pk' | undefined} iconCollection Zbiór ikon" fa - FontAwesome, bi - Bootstrap, pk - Polkurier. Domyślny: fa
 * @property {boolean | undefined} closable Czy wyświetlić przycisk zamykania. Domyślnie TRUE
 * @property {boolean | undefined} spinner Czy wyświetlić spinner zamiast ikony
 */

let nextId = 1;

/**
 * @param {string} message Wiadomość do wyświetlenia w powiadomieniu
 * @param {Partial<ToastMessage>} props Parametry do konfiguracji powiadomienia
 * @return {ToastMessage}
 */
export function createToast(message, props = {}) {

    const toast = document.createElement('pk-toast-message');
    toast.id = `pk-toast-${nextId++}`;
    toast.innerHTML = message;
    toast.type = props.type || TOAST_TYPE_INFO;
    toast.duration = props.duration || DEFAULT_DURATION;
    toast.actionTitle = props.actionTitle || '';
    toast.actionHandler = props.actionHandler;
    toast.icon = props.icon || '';
    toast.iconType = props.iconType || '';
    toast.iconCollection = props.iconCollection || '';
    toast.closable = props.closable !== false;
    toast.spinner = props.spinner === true;

    getContainer().appendChild(toast);
    toast.addEventListener('hidden', () => {
        getContainer().removeChild(toast);
    });
    return toast;
}
