import {LitElement} from "../libs/lit.dist.js?ver=2.4.4";
import {GenericProvider} from "../providers/generic-provider.js?ver=2.4.4";

/**
 * @see https://lit.dev/docs/composition/controllers/
 */
export class ProviderController {

    /**
     * @type GenericProvider
     */
    provider;

    /**
     * @type LitElement
     */
    host;

    static create(host, provider, options = {}) {
        new ProviderController(host, provider, options)
        return provider;
    }

    /**
     *
     * @param {LitElement} host
     * @param {GenericProvider} provider
     * @param {{load?: Boolean, events?: {[key: string]: function}}} options
     */
    constructor(host, provider, options = {}) {
        if (!provider instanceof GenericProvider) {
            throw 'Provider must extend GenericProvider class'
        }

        if (!host instanceof LitElement) {
            throw 'Host must extend LitElement class'
        }

        this.host = host;
        this.provider = provider;
        this.options = options;
        this.requestUpdate = () => this.host.requestUpdate();
        this.host.addController(this);
    }

    hostUpdate() {
        if (this.options.load !== false && !this.provider.isFetched() && !this.provider.isRequested()) {
            this.provider.load().then();
        }
    }

    hostConnected() {
        this.provider.addEventListener('changestate', this.requestUpdate);
        this.provider.addEventListener('change', this.requestUpdate);
        if (this.options.events) {
            for (let event in this.options.events) {
                this.provider.addEventListener(event, this.options.events[event]);
            }
        }
    }

    hostDisconnected() {
        this.provider.removeEventListener('changestate', this.requestUpdate);
        this.provider.removeEventListener('change', this.requestUpdate);
        if (this.options.events) {
            for (let event in this.options.events) {
                this.provider.removeEventListener(event, this.options.events[event]);
            }
        }
    }

}
