import {ajaxRequest} from "../util/http.js?ver=2.4.4";

const dataStore = {};

/**
 * Uniwersalny provider do wczytywania danych z REST serwera do STORE
 */
export class GenericProvider {

    constructor(props = {}) {
        this.name = props.name;
        this.pk = props.pk || 'id';
        this.endpoint = props.endpoint;
        this.params = props.params;

        if (!dataStore[this.name]) {
            dataStore[this.name] = {
                data: [],
                state: {},
                eventsManager: new EventTarget(),
            };
        }
    }

    get fetchedAt() {
        return this.state.fetchedAt || 0;
    }

    get updatedAt() {
        return this.state.updatedAt || 0;
    }

    /**
     * Zwraca wszystkie obiekty z kolekcji.
     * Wczytuje z serwera, jeśli nie wczytano wcześniej.
     * @return {Promise<Array>}
     */
    async load() {
        if (!this.state.isFetched || this.state.isRequested) {
            let currentRequest = this.state.currentRequest;
            if (!currentRequest) {
                currentRequest = this.fetchAll();
                this.state = {currentRequest};
            }
            const data = await currentRequest;
            this.state = {currentRequest: null};
            return data;
        }
        return this.data;
    }

    preload(data) {
        this.data = this.prepareData(data);
        this.state = {
            isRequested: false,
            isFetched: true,
            fetchedAt: Date.now(),
            updatedAt: Date.now(),
        };
        this.dispatchEvent(new CustomEvent('load'));
        return this.data;
    }

    /**
     * Wczytuje dane z serwera.
     * @return {Promise<Array>}
     */
    async reload() {
        return this.fetchAll();
    }

    /**
     * Zwraca wszystkie elementy kolekcji
     * [!!] NIE wczytuje kolekcji z serwera
     * @return {Array}
     */
    all() {
        return this.data;
    }

    /**
     * Zwraca obiekt z kolekcji po kluczu głównym
     * [!!] NIE wczytuje kolekcji z serwera
     *
     * @param pkValue
     */
    get(pkValue) {
        if (pkValue) {
            return this.find(el => el[this.pk] === pkValue);
        }
    }

    find(filter) {
        for (let element of this.data) {
            if (filter(element)) {
                return element;
            }
        }
    }

    filter(filter) {
        return this.data.filter(filter);
    }

    first() {
        return this.data[0];
    }

    /**
     * Wczytuje WSZYSTKIE obiekty z końcówki API
     * @return {Promise<[]>}
     */
    async fetchAll() {
        this.state = {
            isRequested: true,
            requestedAt: Date.now(),
        };

        let data = [];
        try {
            data = await ajaxRequest(this.endpoint, this.params);
            if (!Array.isArray(data)) {
                data = [data];
            }
            this.data = this.prepareData(data);
            this.state = {
                isRequested: false,
                isFetched: true,
                fetchedAt: Date.now(),
                updatedAt: Date.now(),
            };
            this.dispatchEvent(new CustomEvent('load'));
        } catch (error) {
            this.state = {isRequested: false};
            throw error;
        }
        return data;
    }

    /**
     * Dodaje obiektu do kolekcji (np. po zapisie do bazy)
     * @param {Array} data
     */
    add(data) {
        if (!Array.isArray(data)) {
            data = [data];
        }
        this.data = [
            ...this.data,
            ...this.prepareData(data),
        ];
        this.state = {updatedAt: Date.now()};
        return this;
    }

    /**
     * usuwa obiekt do kolekcji (np. po usunięciu do bazy)
     * @param {Array} item
     */
    removeElement(item) {
        this.data = this.data.filter(m => m[this.pk] !== item[this.pk]);
        this.state = {updatedAt: Date.now()};
        return this;
    }

    /**
     * @param {array} data
     * @returns {array}
     */
    prepareData(data) {
        return data;
    }

    set state(props) {
        dataStore[this.name] = {
            ...dataStore[this.name],
            state: {
                ...dataStore[this.name].state,
                ...props,
            }
        };
        this.dispatchEvent(new CustomEvent('changestate'));
    }

    get state() {
        return this.store.state;
    }

    set data(data) {
        dataStore[this.name] = {
            ...dataStore[this.name],
            data,
        };
        this.dispatchEvent(new CustomEvent('change'));
    }

    get data() {
        return this.store.data;
    }

    get store() {
        return dataStore[this.name];
    }

    get length() {
        return this.data.length;
    }

    /**
     * @returns {boolean}
     */
    isRequested() {
        return this.state.isRequested === true;
    }

    /**
     * @returns {boolean}
     */
    isFetched() {
        return this.state.isFetched === true;
    }

    dispatchEvent(type) {
        this.store.eventsManager.dispatchEvent(type);
    }

    addEventListener(type, listener, options) {
        this.store.eventsManager.addEventListener(type, listener, options);
    }

    removeEventListener(type, listener) {
        this.store.eventsManager.removeEventListener(type, listener);
    }

}
