<?php

namespace Polkurier;

final class ConfigManager
{

    private static ?array $envVariables = null;
    private const PATHS = [
        Polkurier::DIR . DIRECTORY_SEPARATOR . 'env.php',
        Polkurier::DIR . DIRECTORY_SEPARATOR . 'env.local.php',
    ];

    public function __construct()
    {
        $this->loadEnvironment();
    }

    private function loadEnvironment(): void
    {
        if (self::$envVariables !== null) {
            return;
        }
        $data = [];
        foreach (self::PATHS as $path) {
            if (file_exists($path)) {
                $data[] = include $path;
            }
        }
        self::$envVariables = array_merge(...$data);
    }

    public function get(string $key, $default = null)
    {
        return self::$envVariables[$key] ?? get_option($key, $default);
    }

    public function getString(string $key, string $default = ''): string
    {
        return (string)$this->get($key, $default);
    }

    public function getInt(string $key, int $default = 0): int
    {
        return (int)$this->get($key, $default);
    }

    public function getFloat(string $key, float $default = 0.0): float
    {
        return (float)$this->get($key, $default);
    }

    public function getArray(string $key, array $default = []): array
    {
        return (array)$this->get($key, $default);
    }

    public function getBoolean(string $key, bool $default = false): bool
    {
        return (bool)filter_var($this->get($key, $default), FILTER_VALIDATE_BOOLEAN);
    }

    public function set(string $key, $value): self
    {
        if (isset(self::$envVariables[$key])) {
            self::$envVariables[$key] = $value;
        } else {
            update_option($key, $value);
        }
        return $this;
    }

}
