<?php
declare(strict_types=1);

namespace Polkurier\Environment;

use Polkurier\ConfigManager;
use Polkurier\Controller\AccountInfoController;
use Polkurier\Controller\AddressesController;
use Polkurier\Controller\CourierPickupController;
use Polkurier\Controller\CourierPointController;
use Polkurier\Controller\CouriersController;
use Polkurier\Controller\MapController;
use Polkurier\Controller\OrderController;
use Polkurier\Controller\PackageCreateController;
use Polkurier\Controller\PackagesDocumentsController;
use Polkurier\Controller\PackagesListController;
use Polkurier\Controller\SettingsAddressesController;
use Polkurier\Controller\SettingsBankAccountsController;
use Polkurier\Controller\SettingsController;
use Polkurier\Controller\SettingsShipmentTemplatesController;
use Polkurier\Controller\ShopOrderController;
use Polkurier\Controller\TemplateController;
use Polkurier\Http\AjaxRouter;
use Polkurier\Http\PageRouter;
use Polkurier\Http\Request;
use Polkurier\Order\OrderManager;
use Polkurier\Order\OrderShippingPointManager;
use Polkurier\Polkurier;
use Polkurier\RequirementsValidator;
use Polkurier\Types\ConfigType;
use Polkurier\Util\Notice;
use Polkurier\Util\OrderTools;
use Polkurier\View;
use WC_Order;
use WP_Post;

class AdminEnvironment extends AbstractEnvironment
{

    private ConfigManager $config;

    public function __construct()
    {
        $this->config = new ConfigManager();
    }

    public function initialize(): void
    {
        add_action('plugins_loaded', function(): void {
            if (RequirementsValidator::isWooCommerceValid()) {
                $this->initializeRouting();
                $this->initializeHooks();
                $this->initSettings();
                $this->enqueueAssets();
                $this->initializeMenus();
            }
        });
    }

    private function initializeRouting(): void
    {
        (new PageRouter())
            ->registerHandler('/polkurier/check-update', SettingsController::class, 'checkUpdateAction')
            ->registerHandler('/polkurier', SettingsController::class, 'authorizationAction')
            ->registerHandler('/polkurier/authorization', SettingsController::class, 'authorizationAction')
            ->registerHandler('/polkurier/addresses', SettingsAddressesController::class, 'senderAddressesAction')
            ->registerHandler('/polkurier/cover_addresses', SettingsAddressesController::class, 'coverAddressesAction')
            ->registerHandler('/polkurier/bank_accounts', SettingsBankAccountsController::class)
            ->registerHandler('/polkurier/templates', SettingsShipmentTemplatesController::class)
            ->registerHandler('/polkurier/carrier', SettingsController::class, 'defaultCourierAction')
            ->registerHandler('/polkurier/couriers_points', SettingsController::class, 'couriersPointsAction')
            ->registerHandler('/polkurier/default_services', SettingsController::class, 'defaultServicesAction')
            ->registerHandler('/polkurier/other', SettingsController::class, 'otherAction')
            ->registerHandler('/polkurier_packages', PackagesListController::class, 'listAction')
            ->registerHandler('/polkurier_packages/list', PackagesListController::class, 'listAction')
            ->registerHandler('/polkurier_packages/details', PackagesListController::class, 'detailsAction')
            ->registerHandler('/polkurier_packages/cancel', PackagesListController::class, 'cancelAction')
            ->registerHandler('/polkurier_packages/label', PackagesDocumentsController::class, 'labelAction')
            ->registerHandler('/polkurier_packages/protocol', PackagesDocumentsController::class, 'protocolAction')
            ->registerHandler('/polkurier_packages/create', PackageCreateController::class, 'createAction');

        (new AjaxRouter())
            ->registerHandler('/polkurier/package_templates', TemplateController::class)
            ->registerHandler('/polkurier/orders/evaluate', OrderController::class, 'evaluateV2Action')
            ->registerHandler('/polkurier/orders/evaluate_v0', OrderController::class, 'evaluateAction')
            ->registerHandler('/polkurier/orders/crete', OrderController::class, 'createAction')
            ->registerHandler('/polkurier/shop_orders', ShopOrderController::class)
            ->registerHandler('/polkurier/couriers', CouriersController::class)
            ->registerHandler('/polkurier/couriers/messages', CouriersController::class, 'messagesAction')
            ->registerHandler('/polkurier/couriers/pickup_date', CourierPickupController::class)
            ->registerHandler('/polkurier/couriers/pickup_hours', CourierPickupController::class, 'pickupHoursAction')
            ->registerHandler('/polkurier/pickup_points/set_default', CourierPointController::class, 'setCourierPoint')
            ->registerHandler('/polkurier/pickup_points/search', CourierPointController::class, 'searchAction')
            ->registerHandler('/polkurier/addresses', AddressesController::class)
            ->registerHandler('/polkurier/account_info', AccountInfoController::class)
            ->registerHandler('/polkurier/map/token', MapController::class, 'getMapTokenAction');
    }

    private function initializeHooks(): void
    {

        add_action('woocommerce_admin_order_data_after_shipping_address', function ($order): void {
            echo '<div class="address">';
            $shippingPoint = (new OrderShippingPointManager())->get($order);
            $shippingMethod = OrderTools::getShippingMethod($order);
            if ($shippingPoint !== null) {
                echo '<p>'
                    . '<strong>' . ($shippingMethod !== null && OrderTools::isOwnShippingMethod($shippingMethod) ? $shippingMethod->getSelectedPointLabel() : 'Wybrany punkt odbioru') . ':</strong><br>'
                    . '<strong><u>' . $shippingPoint->code . '</u></strong><br>'
                    . nl2br($shippingPoint->label)
                    . '</p>';
            }
            echo '</div>';
        }, 0, 1);

        add_filter("plugin_action_links_" . Polkurier::SLUG, function ($links) {
            $links[] = '<a href="admin.php?page=polkurier">Ustawienia</a>';
            $links[] = '<a href="admin.php?page=polkurier&tab=check-update">Sprawdź aktualizację</a>';
            return $links;
        });

        add_filter('woocommerce_admin_order_actions', function (array $actions, $order): array {
            $orderStatus = $order->get_status();
            if ($orderStatus !== 'pending' && $orderStatus !== 'on-hold' && $orderStatus !== 'completed' && $orderStatus !== 'cancelled' && $orderStatus !== 'refunded' && $orderStatus !== 'failed') {
                $order_id = method_exists($order, 'get_id') ? $order->get_id() : $order->id;
                $order = (new OrderManager())->getByShopOrderId($order_id);
                if (!$order) {
                    $shipButtonUrl = wp_nonce_url(admin_url('admin.php?page=polkurier_packages&action=create&order_id=' . $order_id));
                    $actions['polkurier'] = array(
                        'url' => wp_nonce_url($shipButtonUrl, 'polkurier'),
                        'name' => __('Zamów usługę Polkurier', 'woocommerce'),
                        'action' => 'polkurier polkurier-action-button',
                    );
                    $actions['polkurier-quick'] = array(
                        'url' => $shipButtonUrl,
                        'name' => __('Szybie zamówienie przesyłki', 'woocommerce'),
                        'action' => 'polkurier-quick polkurier-action-button polkurier-prevent-default',
                    );
                } else {
                    $actions['polkurier-print-label'] = array(
                        'url' => wp_nonce_url(admin_url('admin.php?page=polkurier_packages&action=label&label=' . $order->getOrderNumber())),
                        'name' => __('Drukuj etykietę', 'woocommerce'),
                        'action' => 'polkurier wc-action-button-polkurier-print-label polkurier-action-button polkurier-open-in-new-tab',
                    );
                    $actions['polkurier-print-protocol'] = array(
                        'url' => wp_nonce_url(admin_url('admin.php?page=polkurier_packages&action=protocol&label=' . $order->getOrderNumber())),
                        'name' => __('Drukuj protokół', 'woocommerce'),
                        'action' => 'polkurier wc-action-button-polkurier-print-protocol polkurier-action-button polkurier-open-in-new-tab',
                    );
                }
            }
            return $actions;
        }, 100, 2);

        $trackingOrderColumn = static function (string $column_name, $shopOrder = null) {
            global $post;

            if ($shopOrder instanceof WC_Order) {
                $orderId = $shopOrder->get_id();
            } else if ($post !== null) {
                $orderId = $post->ID;
            } else {
                $orderId = null;
            }

            if ($column_name === 'package_number') {
                echo View::render('OrdersList/package_number_column', [
                    'shopOrderId' => $orderId,
                    'order' => $orderId ? (new OrderManager())->getByShopOrderId($orderId) : null,
                ]);
            }
        };

        add_filter('manage_shop_order_posts_custom_column', $trackingOrderColumn);
        add_action('manage_woocommerce_page_wc-orders_custom_column', $trackingOrderColumn, 10, 2);

        add_action('add_meta_boxes', static function (): void {
            add_meta_box(Polkurier::NAME . '_delivery', 'Polkurier.pl', static function ($post): void {
                $shopOrder = null;
                if ($post instanceof WC_Order) {
                    $shopOrder = wc_get_order($post->get_id());
                } else if ($post instanceof WP_Post) {
                    $shopOrder = wc_get_order($post->ID);
                }
                echo View::render('OrderView/metabox', [
                    'shopOrderId' => $shopOrder ? $shopOrder->get_id() : null,
                    'order' => $shopOrder ? (new OrderManager())->getByShopOrderId($shopOrder->get_id()) : null,
                    'nonce' => wp_create_nonce('polkurier-nonce'),
                ]);
            }, wc_get_page_screen_id('shop-order'), 'side', 'high');
        });

        add_action('all_admin_notices', function (): void {
            if (!RequirementsValidator::isWooCommerceValid()) {
                Notice::error('Błąd Polkurier: nie znaleziono wtyczki WooCommerce. Sprawdź konfigurację swojej instancji Wordpress.');
                return;
            }

            $screen = get_current_screen();
            if ($screen !== null && $screen->id === wc_get_page_screen_id('shop-order')) {
                echo View::render('OrdersList/column_notice');
            }

            if (empty($this->config->getString(ConfigType::ACCOUNT_ID)) || empty($this->config->getString(ConfigType::APIKEY))) {
                add_action('admin_notices', array($this, 'no_settings_notice'));
                Notice::error('Aby użyć wtyczki Polkurier, wprowadź wszystkie <a href="admin.php?page=polkurier">ustawienia</a>.');
            }
        });

    }

    private function initSettings(): void
    {
        add_action('admin_init', function (): void {

            $renderInput = static function (string $name, string $type = 'text'): void {
                echo '<input type="' . $type . '" name="' . $name . '" id="' . $name . '" value="' . get_option($name) . '" >';
            };

            add_settings_section(
                'polkurier_authorization',
                __('Autoryzacja', 'polkurier'),
                static function (): void {
                    echo __('Ustawienia autoryzacji', 'polkurier');
                },
                'polkurier_authorization'
            );
            add_settings_field(
                'polkurier_account_id',
                __('ID konta', 'polkurier'),
                static function () use ($renderInput): void {
                    $renderInput('polkurier_account_id');
                },
                'polkurier_authorization',
                'polkurier_authorization',
                array('label_for' => 'polkurier_account_id')
            );
            add_settings_field(
                'polkurier_apikey',
                __('Token', 'polkurier'),
                static function () use ($renderInput): void {
                    $renderInput('polkurier_apikey');
                },
                'polkurier_authorization',
                'polkurier_authorization',
                array('label_for' => 'polkurier_apikey')
            );
            register_setting('polkurier_authorization', 'polkurier_apikey', ['type' => 'string']);
            register_setting('polkurier_authorization', 'polkurier_account_id', ['type' => 'string']);
        });
    }

    private function enqueueAssets(): void
    {
        add_filter('wp_script_attributes', function (array $attributes): array {
            if (isset($attributes['id']) && stripos($attributes['id'], '--module') !== false) {
                $attributes['type'] = 'module';
            }
            return $attributes;
        }, 1, 1);

        add_action('admin_head', function () {
            echo View::render('scripts/admin_global_js', [
                'ajaxNonce' => wp_create_nonce('polkurier-nonce'),
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'esmUrl' => Polkurier::URL,
            ]);
        });

        add_action('admin_enqueue_scripts', function (): void {
            wp_enqueue_script('pk-orders-list--module', Polkurier::URL . 'admin/js/OrdersList/index.js?ver=2.4.4', [], Polkurier::VERSION);

            wp_enqueue_style('pk-select2', Polkurier::URL . 'admin/css/select2.min.css', [], Polkurier::VERSION);
            wp_enqueue_style('pk-utils', Polkurier::URL . 'admin/css/pk-utils.css', [], Polkurier::VERSION);
            wp_enqueue_style('pk-admin', Polkurier::URL . 'admin/css/polkurier-admin.css', [], Polkurier::VERSION);
            wp_enqueue_style('pk-map', 'https://maps.polkurier.pl/assets/dist/points-map.css', [], Polkurier::VERSION);

            wp_enqueue_script('pk-admin', Polkurier::URL . 'admin/js/polkurier-admin.js?ver=2.4.4', ['jquery'], Polkurier::VERSION);
            wp_enqueue_script('pk-map', 'https://maps.polkurier.pl/assets/dist/points-map.bundle.js?ver=2.4.4', [], Polkurier::VERSION, [
                'in_footer' => true,
            ]);
        });
    }

    private function initializeMenus(): void
    {
        $doRoutingAction = static function (string $actionProp = 'action'): void {
            $request = new Request();
            $path = rtrim('/' . $request->query->getString('page') . '/' . $request->query->getString($actionProp), '/');
            do_action("polkurier_page_routing_$path");
        };

        add_action('admin_menu', function () use ($doRoutingAction): void {
            add_submenu_page(
                'woocommerce',
                'Ustawienia Polkurier',
                'Ustawienia Polkurier',
                'manage_options',
                'polkurier',
                static function () use ($doRoutingAction): void {
                    $doRoutingAction('tab');
                }
            );

            add_submenu_page(
                'woocommerce',
                'Przesyłki Polkurier',
                'Przesyłki Polkurier',
                'manage_woocommerce',
                'polkurier_packages',
                static function () use ($doRoutingAction): void {
                    $doRoutingAction();
                }
            );
        });
    }

}
