<?php
declare(strict_types=1);

namespace Polkurier\Environment;

use Polkurier\Integrations\FlexibleShipping;
use Polkurier\Order\OrderShippingPointManager;
use Polkurier\Order\ShippingPoint;
use Polkurier\Polkurier;
use Polkurier\RequirementsValidator;
use Polkurier\ShippingMethods\AbstractShippingMethod;
use Polkurier\Types\CheckoutPointButtonPosition;
use Polkurier\Util\Arr;
use Polkurier\View;
use Throwable;
use WC_Order;
use WC_Shipping_Rate;

class FrontEnvironment extends AbstractEnvironment
{

    private Polkurier $polkurier;

    public function __construct(Polkurier $polkurier)
    {
        $this->polkurier = $polkurier;
    }

    public function initialize(): void
    {
        add_action('plugins_loaded', function(): void {
            if (RequirementsValidator::isWooCommerceValid()) {
                $this->initializeAssets();
                $this->initializeCheckoutHooks();
            }
        });
    }

    private function initializeAssets(): void
    {
        add_action('wp_enqueue_scripts', function () {
            if (is_checkout()) {
                wp_enqueue_script('pk-checkout-page', Polkurier::URL . 'public/js/polkurier-checkout-page.js?ver=2.4.4', ['jquery'], Polkurier::URL, true);
                wp_enqueue_script('pk-points-map', 'https://maps.polkurier.pl/assets/dist/points-map.bundle.js?ver=2.4.4', ['jquery'], Polkurier::URL, true);
                wp_enqueue_style('pk-points-map', 'https://maps.polkurier.pl/assets/dist/points-map.css', [], Polkurier::URL);
            }
        });
    }

    private function initializeCheckoutHooks(): void
    {

        // Numer telefonu i email obowiązkowe
        add_filter('woocommerce_checkout_fields', function (array $fields): array {

            if (isset($fields['billing']['billing_phone'])) {
                $fields['billing']['billing_phone']['required'] = true;
            }

            if (isset($fields['billing']['billing_email'])) {
                $fields['billing']['billing_email']['required'] = true;
            }
            return $fields;
        });

        add_filter('woocommerce_get_order_item_totals', function(array $items, $wcOrder) {
            $point = (new OrderShippingPointManager())->get($wcOrder);
            if (isset($items['shipping']) && $point !== null) {
                $items['shipping']['value'] .= View::render('checkout/get_order_item_totals_point', [
                    'point' => $point,
                    'pointId' => $point->code,
                    'pointLabel' => $point->label,
                ]);
            }
            return $items;
        }, 2, 100);

        // Formatowanie adresu wysyłki po złożeniu zamówienia
        add_filter('woocommerce_order_get_formatted_shipping_address', function ($addressData) {
            global $wp;
            $orderId = (int)Arr::get($wp->query_vars, 'order-received');
            if ($orderId > 0) {
                $order = new WC_Order($orderId);
                $orderShippingMethod = Arr::get(array_values($order->get_shipping_methods()), '0');
                $shippingMethod = Polkurier::instance()->getShippingMethodById($orderShippingMethod->get_method_id(), (int)$orderShippingMethod->get_instance_id());
                if ($shippingMethod !== null) {
                    $shouldShowPickupPointAddress = false;
                    if ($shippingMethod instanceof AbstractShippingMethod && $shippingMethod->isParcelPickupPoint()) {
                        $shouldShowPickupPointAddress = true;
                    }

                    foreach ($this->polkurier->getIntegrations() as $integration) {
                        if ($integration instanceof FlexibleShipping && $integration->shouldShowMapButton($shippingMethod)) {
                            $shouldShowPickupPointAddress = true;
                        }
                    }

                    $point = (new OrderShippingPointManager())->get($order);
                    if ($shouldShowPickupPointAddress && $point !== null) {
                        return View::render('checkout/order_get_formatted_shipping_address', [
                            'pointId' => $point->code,
                            'pointLabel' => $point->label,
                            'provider' => $point->provider,
                            'pointType' => $point->type,
                            'point' => $point,
                        ]);
                    }
                }
            }
            return $addressData;
        });

        $renderMapButton = function ($method, string  $position): void {
            if (is_checkout() && get_option('polkurier_layout_checkout_point_button_position') === $position) {
                $currentMethod = Polkurier::instance()->getCurrentShippingMethod();
                if ($method instanceof WC_Shipping_Rate && $currentMethod !== null) {
                    $isValid = $method->id === $currentMethod->id . ':' . $currentMethod->get_instance_id();
                } else {
                    $isValid = true;
                }
                if ($currentMethod !== null && $isValid) {
                    if ($currentMethod instanceof AbstractShippingMethod && $currentMethod->isParcelPickupPoint()) {
                        $currentMethod->render();
                    }
                    foreach ($this->polkurier->getIntegrations() as $integration) {
                        if ($integration instanceof FlexibleShipping && $integration->shouldShowMapButton($currentMethod)) {
                            $integration->renderShippingMethod($currentMethod);
                        }
                    }
                }
            }
        };

        // Wyświetla przycisk "Wybierz paczkomat" pod blokiem wyboru metod wysyłki
        add_action('woocommerce_review_order_after_shipping', function () use ($renderMapButton): void {
            $renderMapButton(null, CheckoutPointButtonPosition::AFTER_SHIPPING_BLOCK);
        });

        // Wyświetla przycisk "Wybierz paczkomat" w wybranym miejscu
        foreach (CheckoutPointButtonPosition::VALID_POSITIONS as $positionName) {
            $filterName = $positionName;
            if (!empty(CheckoutPointButtonPosition::MAP_POSITIONS[$positionName])) {
                $filterName = CheckoutPointButtonPosition::MAP_POSITIONS[$positionName];
            }

            $priority = CheckoutPointButtonPosition::DEFAULT_PRIORITY;
            if (!empty(CheckoutPointButtonPosition::PRIORITIES[$positionName])) {
                $priority = CheckoutPointButtonPosition::PRIORITIES[$positionName];
            }

            add_action($filterName, function ($method) use ($renderMapButton, $positionName): void {
                $renderMapButton($method, $positionName);
            }, $priority);
        }

        add_action('woocommerce_checkout_update_order_review', function (): void {
            $currentMethod = $this->polkurier->getCurrentShippingMethod();
            if ($currentMethod instanceof AbstractShippingMethod) {
                $currentMethod->onCheckoutUpdateOrderReview();
            }
        });

        // Walidacja zamówienia legacy
        add_action('woocommerce_checkout_process', function (): void {
            $currentMethod = Polkurier::instance()->getCurrentShippingMethod();
            if ($currentMethod instanceof AbstractShippingMethod && $currentMethod->isParcelPickupPoint()) {
                $currentMethod->onCheckoutProcess();
            }
            foreach ($this->polkurier->getIntegrations() as $integration) {
                if ($integration instanceof FlexibleShipping) {
                    $integration->onCheckoutProcess();
                }
            }
        });

        // Walidacja zamówienia Blocks
        add_action('woocommerce_store_api_checkout_order_processed', function (WC_Order $order): void {
            $currentMethod = Polkurier::instance()->getCurrentShippingMethod();
            if ($currentMethod instanceof AbstractShippingMethod && $currentMethod->isParcelPickupPoint()) {
                $currentMethod->onCheckoutProcess($order);
            }
            foreach ($this->polkurier->getIntegrations() as $integration) {
                if ($integration instanceof FlexibleShipping) {
                    $integration->onCheckoutProcess($order);
                }
            }
        }, 10, 3);


        $saveOrderDeliveryPoint = static function (WC_Order $order): void {
            global $wpdb;

            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
                return;
            }

            $logger = wc_get_logger();
            if ((new OrderShippingPointManager())->get($order) !== null) {
                if ($logger !== null) {
                    $logger->warning("Już jest punkt do zamówienia {$order->get_id()}");
                }
                return;
            }

            $sessionStore = WC()->session !== null
                ? WC()->session->get(Polkurier::NAME . '_delivery_point')
                : null;

            $point = new ShippingPoint();
            $point->wp_order_id = $order->get_id();
            if (!empty($sessionStore)) {
                $point->code = esc_attr((string)Arr::get($sessionStore, 'id'));
                $point->provider = esc_attr((string)Arr::get($sessionStore, 'provider'));
                $point->label = esc_attr((string)Arr::get($sessionStore, 'label'));
                $point->type = esc_attr((string)Arr::get($sessionStore, 'type'));
            } else {
                $point->code = esc_attr((string)Arr::get($_POST, 'polkurier_point_id'));
                $point->provider = esc_attr((string)Arr::get($_POST, 'polkurier_provider'));
                $point->label = esc_attr((string)Arr::get($_POST, 'polkurier_point_label'));
                $point->type = esc_attr((string)Arr::get($_POST, 'polkurier_point_type'));
            }

            if (!empty($point->code)) {
                $logData = [
                    'VERSION' => Polkurier::VERSION,
                    'DB_VERSION' => (int)get_option('pk_database_version'),
                    '$orderId' => $order->get_id(),
                    '$formData' => $_POST,
                    '$sessionStore' => $sessionStore,
                    '$point' => $point,
                ];

                try {
                    (new OrderShippingPointManager())->save($point);
                } catch (Throwable $e) {
                    if (empty($point->id) && $logger !== null) {
                        $logger->critical('Nieudany Zapis danych punktu odbioru (1052): ' . $e->getMessage(), array_merge($logData, [
                            'exception' => $e->getMessage(),
                            '_db_error' => $wpdb->last_error,
                        ]));
                    }
                }

                if ($logger !== null) {
                    $logger->info('Zapis danych punktu odbioru', array_merge($logData, [
                        '_db_error' => $wpdb->last_error,
                    ]));
                }
            }

            if (WC()->session !== null) {
                WC()->session->set(Polkurier::NAME . '_delivery_point', null);
            }
        };

        // Zapis punktu odbioru blocks
        add_action('woocommerce_store_api_checkout_order_processed', function ($orderId) use ($saveOrderDeliveryPoint): void {
            $saveOrderDeliveryPoint(wc_get_order($orderId));
        });

        // Zapis punktu odbioru legacy
        add_action('woocommerce_checkout_order_created', function ($order) use ($saveOrderDeliveryPoint): void {
            $saveOrderDeliveryPoint($order);
        });

    }

}
