<?php

namespace Polkurier\Http;

use Exception;
use Polkurier\Exception\HttpException;
use Polkurier\Logger;

class AjaxRouter
{

    public function registerHandler(string $path, $handlerOrClassName, string $methodName = 'indexAction'): AjaxRouter
    {
        add_action("wp_ajax_$path", function () use ($handlerOrClassName, $methodName) {
            if (is_callable($handlerOrClassName)) {
                $this->handlerRequest($handlerOrClassName);
            }

            if (is_string($handlerOrClassName) && class_exists($handlerOrClassName)) {
                $this->handlerRequest(function (Request $request) use ($handlerOrClassName, $methodName) {
                    return (new $handlerOrClassName())->$methodName($request);
                });
            }

            if (is_object($handlerOrClassName) && method_exists($handlerOrClassName, $methodName)) {
                $this->handlerRequest(function (Request $request) use ($handlerOrClassName, $methodName) {
                    return $handlerOrClassName->$methodName($request);
                });
            }

            return null;
        });
        return $this;
    }

    public function handlerRequest(callable $action): void
    {
        check_ajax_referer( 'polkurier-nonce' );
        try {
            $response = $action(new Request());
            if (!$response instanceof Response) {
                throw new HttpException('Controller must return an instance of Response class', 500);
            }
        } catch (HttpException $e) {
            if ($e->getPrevious() !== null) {
                (new Logger())->exception($e->getPrevious());
            }
            (new Logger())->exception($e);
            $response = new JsonResponse(['error' => $e->getMessage()], $e->getCode());
        } catch (Exception $e) {
            if ($e->getPrevious() !== null) {
                (new Logger())->exception($e->getPrevious());
            }
            (new Logger())->exception($e);
            $response = new JsonResponse(['error' => $e->getMessage()], 500);
        }
        $response->send();
        exit;
    }

}
