<?php

namespace Polkurier\Normalizer;

use DateTime;
use DateTimeImmutable;
use Polkurier\Util\Dates;
use Polkurier\Util\Strings;
use ReflectionClass;
use ReflectionException;
use ReflectionProperty;
use ReflectionType;

class Denormalizer
{

    private static array $loadedClassesCache = [];

    /**
     * @throws ReflectionException
     */
    public function denormalize(object $object, array $data): object
    {
        $reflection = new ReflectionClass($object);
        foreach ($reflection->getProperties() as $prop) {
            $value = $data[$prop->getName()] ?? null;
            $setter = 'set' . Strings::toPascalCase($prop->getName());
            if (method_exists($object, $setter)) {
                $method = $reflection->getMethod($setter);
                $parameters = $method->getParameters();
                if (isset($parameters[0])) {
                    $type = $parameters[0]->getType();
                    if ($type !== null) {
                        $value = $this->castType($value, $parameters[0]->getType(), $prop, $reflection);
                    }
                }
                $object->$setter($value);
            } else {
                $prop->setAccessible(true);
                $prop->setValue($object, $this->castType($value, $prop->getType(), $prop, $reflection));
            }
        }
        return $object;
    }

    /**
     * @return mixed
     * @throws ReflectionException
     */
    private function castType($value, ?ReflectionType $type, ReflectionProperty $property, ReflectionClass $class)
    {
        if ($type === null) {
            return $value;
        }

        if ($value === null && $type->allowsNull()) {
            return null;
        }

        if ($type->isBuiltin()) {
            switch ($type->getName()) {
                case 'int':
                case 'integer':
                    return (int)$value;

                case 'string':
                    return (string)$value;

                case 'float':
                    return (float)$value;

                case 'bool':
                case 'boolean':
                    return (bool)$value;

                case 'array':
                    if (empty($value)) {
                        return [];
                    }

                    [$TKey, $TVal] = $this->getArrayItemType((string)$property->getDocComment());
                    if ($TVal !== 'mixed') {
                        $className = '';
                        $loadedClasses = $this->getLoadedClassNames($class);

                        if (isset($loadedClasses[$TVal])) {
                            $className = $loadedClasses[$TVal];
                        }

                        if (!class_exists($className)) {
                            $fqcn = $class->getNamespaceName() . '\\' . $TVal;
                            if (
                                $className !== 'bool' &&
                                $className !== 'boolean' &&
                                $className !== 'int' &&
                                $className !== 'integer' &&
                                $className !== 'string' &&
                                $className !== 'float' &&
                                $className !== 'array' &&
                                @class_exists($fqcn)
                            ) {
                                $className = $fqcn;
                            }
                        }

                        if ($className !== '') {
                            foreach ((array)$value as $key => $val) {
                                $value[$key] = $this->castToObject($val, $className);
                            }
                        }
                    }
                    return (array)$value;
            }
        } else if (class_exists($type->getName())) {
            return $this->castToObject($value, $type->getName());
        }

        return $value;
    }

    /**
     * @throws ReflectionException
     */
    private function castToObject($value, string $className): object
    {
        if ($className === DateTimeImmutable::class) {
            return Dates::dateTimeImmutableOrNull($value);
        }

        if ($className === DateTime::class) {
            return Dates::dateTimeOrNull($value);
        }

        $valueRaw = $value;
        $value = (new ReflectionClass($className))->newInstanceWithoutConstructor();
        if (is_array($valueRaw)) {
            $this->denormalize($value, $valueRaw);
        }
        return $value;
    }

    private function getArrayItemType(string $comment): array
    {
        $TKey = 'int';
        $TVal = 'mixed';
        if (strpos($comment, '/**') === 0) {
            $matches = [];
            preg_match('/@var\s+((\w+)\[]|Array<(\w+,\s*\w+)>|Array<(\w+)>)/im', $comment, $matches);
            if (count($matches) === 3) {
                $TVal = $matches[2];
            } else if (count($matches) === 5) {
                $TVal = $matches[4];
            } else if (count($matches) === 4) {
                if (strpos($matches[3], ',') !== false) {
                    [$TKey, $TVal] = explode(',', str_replace(' ', '', $matches[3]));
                } else {
                    $TVal = $matches[3];
                }
            }
        }
        return [$TKey, $TVal];
    }

    private function getLoadedClassNames(ReflectionClass $class): array
    {
        if (!isset(self::$loadedClassesCache[$class->getName()])) {
            $classNames = [];
            $file = file($class->getFileName());
            foreach ($file as $line) {
                if (strpos($line, 'class') === 0) {
                    break;
                }
                if (strpos($line, 'use') === 0) {
                    $use = trim(substr($line, 3), " \n\r\t\v\0;");
                    if (strpos($use, ' ') !== false) {
                        $use = explode(' ', $use)[0];
                    }
                    if (strpos($use, '\\') !== false) {
                        $namespaces = explode('\\', $use);
                        $className = array_pop($namespaces);
                    } else {
                        $className = $use;
                    }
                    $classNames[$className] = $use;
                }
            }
            self::$loadedClassesCache[$class->getName()] = $classNames;
        }
        return self::$loadedClassesCache[$class->getName()];
    }

}
