<?php

namespace Polkurier\Order;

use Polkurier\Logger;
use Polkurier\Normalizer\Denormalizer;
use Polkurier\Normalizer\Normalizer;
use Polkurier\Util\Database;
use ReflectionException;
use Throwable;

class OrderManager
{

    private static array $orderCache = [];

    private static function cacheOrderBy(Order $order, ?string $field = null, $value = null): void
    {
        if ($field !== null && $value !== null) {
            $field = strtolower($field);
            self::$orderCache["by::$field:$value"] = $order;
        }
        if ($order->getId() !== null) {
            self::$orderCache["by::id:{$order->getId()}"] = $order;
        }
    }

    private static function getCachedBy(string $field, $value): ?Order
    {
        $field = strtolower($field);
        $key = "by::$field:$value";
        return self::$orderCache[$key] ?? null;
    }

    public function getByShopOrderId(int $id): ?Order
    {
        return $this->getOrderBy('shop_order_id', $id);
    }

    public function getOrderById(string $id): ?Order
    {
        return $this->getOrderBy('id', (int)$id);
    }

    public function getOrderByNumber(string $orderNumber): ?Order
    {
        return $this->getOrderBy('order_number', $orderNumber);
    }

    /**
     * @throws ReflectionException
     */
    private function normalizeForStorage(Order $order): array
    {
        $data = (new Normalizer())->normalize($order);
        foreach ($data as $key => $value) {
            if (is_array($value) || is_object($value)) {
                $data[$key] = json_encode($value);
            }
        }
        return $data;
    }

    /**
     * @throws ReflectionException
     */
    private function denormalizeFromStorage(Order $order, array $data): void
    {
        foreach ($data as $key => $value) {
            if (is_string($value) && $value !== '' && ($value[0] === '[' || $value[0] === '{')) {
                $data[$key] = json_decode($value, true);
            }
        }

        $denormalizer = new Denormalizer();
        $denormalizer->denormalize($order, $data);

        if (isset($data['pobranie']) && $data['pobranie'] > 0 && $order->getCod() === null) {
            $order->setCod(new OrderCod((float)$data['pobranie'], '', '', ''));
        }
    }

    public function save(Order $order): OrderManager
    {
        global $wpdb;
        try {
            if ($order->getId() !== null) {
                $wpdb->update($wpdb->prefix . 'polkurier_orders', $this->normalizeForStorage($order), [
                    'id' => $order->getId()
                ]);
            } else {
                $id = $wpdb->insert($wpdb->prefix . 'polkurier_orders', $this->normalizeForStorage($order));
                $order->setShopOrderId($id);
                self::cacheOrderBy($order);
            }
            if ($wpdb->last_error !== '') {
                (new Logger())->error($wpdb->last_error, $this->normalizeForStorage($order));
            }
        } catch (Throwable $e) {
            (new Logger())->exception($e);
        }
        return $this;
    }

    public function getOrderBy(string $field, $value): ?Order
    {
        global $wpdb;

        $order = self::getCachedBy($field, $value);
        if ($order) {
            return $order;
        }

        try {
            $data = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}polkurier_orders WHERE `$field` = %s", $value), ARRAY_A);
            if (!empty($data)) {
                $order = new Order();
                $this->denormalizeFromStorage($order, $data);
                self::cacheOrderBy($order, $field, $value);
                return $order;
            }
        } catch (Throwable $e) {
            (new Logger())->exception($e);
        }
        return null;
    }

    public function hasOrderDefined(int $id): bool
    {
        return $this->getByShopOrderId($id) !== null;
    }

    private function createWhere(array $where): string
    {
        $searchQuery = '';
        if (isset($where['searchQuery'])) {
            $searchQuery = trim($where['searchQuery']);
            unset($where['searchQuery']);
        }

        $query = Database::createWhere($where);
        if (!empty($searchQuery)) {
            if (!empty($query)) {
                $query = "($query) AND ";
            }
            $query .= Database::createSearchWhere($searchQuery, array(
                'order_number',
                'label',
                'address',
                'address_to',
            ));
        }
        return $query;
    }

    /**
     * @return Order[]
     * @throws ReflectionException
     */
    public function getAll(array $where = [], $order = null, ?int $limit = null, ?int $offset = null): array
    {
        global $wpdb;
        $data = [];
        $query = "SELECT * FROM {$wpdb->prefix}polkurier_orders";

        $whereString = $this->createWhere($where);
        if (!empty($whereString)) {
            $query .= ' WHERE ' . $whereString;
        }

        if (!empty($order)) {
            $col = is_array($order) ? $order[0] : $order;
            $dir = strtoupper(is_array($order) ? $order[1] : 'ASC');
            $dir = $dir === 'DESC' ? 'DESC' : 'ASC';
            $query .= " ORDER BY `$col` $dir ";
        }

        $limit = (int)$limit;
        $offset = (int)$offset;
        if (!empty($limit)) {
            $query .= ' LIMIT ' . $limit;
            if (!empty($offset)) {
                $query .= ' OFFSET ' . $offset;
            }
        }

        foreach ($wpdb->get_results($query, ARRAY_A) as $row) {
            $polkurierOrder = new Order();
            $this->denormalizeFromStorage($polkurierOrder, $row);
            self::cacheOrderBy($polkurierOrder);
            $data[] = $polkurierOrder;
        }

        return $data;
    }

    public function countAll(array $where = []): int
    {
        global $wpdb;
        $query = "SELECT COUNT(*) as c FROM {$wpdb->prefix}polkurier_orders";
        $whereString = $this->createWhere($where);
        if (!empty($whereString)) {
            $query .= ' WHERE ' . $whereString;
        }
        $data = $wpdb->get_results($query, ARRAY_A);
        return (int)$data[0]['c'];
    }

    public function deleteByOrderNumber(string $orderNumber): void
    {
        global $wpdb;
        $wpdb->delete($wpdb->prefix . 'polkurier_orders', ['order_number' => $orderNumber]);
    }

}
