<?php

namespace Polkurier;

use DateInterval;
use DateTime;
use Exception;
use Polkurier\Exception\ApiException;
use Polkurier\Order\Order;
use Polkurier\Order\OrderManager;
use Polkurier\Util\Dates;
use ReflectionException;

class OrderStatusUpdater
{

    /**
     * Co ile sekund sprawdzać status
     */
    private static int $UPDATE_STATUS_INTERVAL = 1440; // 4H

    /**
     * Przez ile dni po utworzeniu zamówienia sprawdzać status
     */
    private static int $UPDATE_STATUS_DAYS = 21;

    private ApiClient $api;
    private OrderManager $ordersManager;

    public function __construct()
    {
        $this->api = new ApiClient();
        $this->ordersManager = new OrderManager();
    }

    /**
     * @throws ApiException
     */
    public function getOrderStatus(string $orderNumber): array
    {
        return $this->api->makeApiRequest('get_status', [
            'orderno' => $orderNumber,
        ]);
    }

    /**
     * @throws Exception
     */
    public function shouldUpdateOrderStatus(Order $polkurierOrder): bool
    {
        $lastStatusUpdateTime = (int)($polkurierOrder->getExtra()['lastStatusUpdateTime'] ?? 0);
        $dateTo = (clone $polkurierOrder->getDateCreated());
        $dateTo->add(new DateInterval('P' . self::$UPDATE_STATUS_DAYS . 'D'));
        return
            !empty($polkurierOrder->getOrderNumber()) &&
            $lastStatusUpdateTime + self::$UPDATE_STATUS_INTERVAL < time() &&
            $dateTo >= new DateTime() &&
            in_array($polkurierOrder->getStatusCode(), ['', 'O', 'OC', 'P', 'WP', 'PZ', 'Z']);
    }

    /**
     * @param Order $polkurierOrder
     */
    public function updateOrderStatus(Order $polkurierOrder): void
    {
        if (!empty($polkurierOrder->getOrderNumber())) {
            try {
                $polkurierStatus = $this->getOrderStatus($polkurierOrder->getOrderNumber());
                if (!empty($polkurierStatus)) {
                    $polkurierOrder->setStatusCode((string)($polkurierStatus['status_code'] ?? ''));
                    $polkurierOrder->setExtraValue('status_date', $polkurierStatus['status_date'] ?? null);
                    $polkurierOrder->setDeliveredDate(
                        $polkurierOrder->getStatusCode() === 'D'
                            ? Dates::dateTimeImmutableOrNull($polkurierStatus['delivered_date'] ?? null)
                            : null
                    );
                    $polkurierOrder->setExtraValue('lastStatusUpdateSuccess', true);
                } else {
                    $polkurierOrder->setExtraValue('lastStatusUpdateSuccess', false);
                }
                $polkurierOrder->setExtraValue('lastStatusUpdateTime', time());
                $this->ordersManager->save($polkurierOrder);
            } catch (Exception $e) {
            }
        }
    }

    /**
     * Aktualizuje wszystkie statusy
     * @throws ReflectionException
     */
    public function updateAllOrdersStatuses(): void
    {
        $orders = $this->ordersManager->getAll(["status_code IN ('', 'O', 'OC', 'P', 'WP', 'PZ', 'Z')" => null]);
        foreach ($orders as $order) {
            $this->updateOrderStatus($order);
        }
    }

}
