<?php

namespace Polkurier\WPTables;

use Exception;
use Polkurier\CouriersManager;
use Polkurier\Order\Order;
use Polkurier\Order\OrderAddress;
use Polkurier\Order\OrderManager;
use Polkurier\OrderStatusUpdater;
use Polkurier\Types\OrderStatusType;
use ReflectionException;
use WP_List_Table;


class OrdersListTable extends WP_List_Table
{

    private static int $DEFAULT_PAGE_SIZE = 50;
    private static int $MAX_DEFAULT_PAGE_SIZE = 250;
    private CouriersManager $couriersManager;
    private OrderManager $ordersManager;

    public function __construct($args = [])
    {
        parent::__construct($args);
        $this->couriersManager = new CouriersManager();
        $this->ordersManager = new OrderManager();
    }

    public function get_columns(): array
    {
        return [
            'mark' => 'Zaznacz',
            'order_number' => 'Numer zamówienia',
            'address' => 'Adres nadawcy',
            'address_to' => 'Adres odbiorcy',
            'date_send' => 'Data nadania',
            'label' => 'Listy przewoz.',
            'status_date' => 'Zmiana statusu',
            'status' => 'Status',
            'delivered_date' => 'Data doręczenia',
            'price' => 'Cena netto/brutto',
            'pobranie' => 'Pobranie',
            'user' => 'Użytkownik',
            'action' => 'Działania',
        ];
    }

    public function get_sortable_columns(): array
    {
        return [
            'order_number' => array('id', false),
            'status_date' => array('status_date', false),
            'status' => array('status', false),
            'delivered_date' => array('delivered_date', false),
            'price_net' => array('price', false),
        ];
    }

    /**
     * @throws ReflectionException
     */
    public function prepare_items($unpaid_only = false): void
    {
        $columns = $this->get_columns();
        $hidden = array('_id');
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = array($columns, $hidden, $sortable);

        $orderBy = (!empty($_GET['orderby'])) ? $_GET['orderby'] : 'id';
        $orderDir = (!empty($_GET['order'])) ? $_GET['order'] : 'DESC';

        $currentPage = $this->get_pagenum();
        $pageSize = (int)(!empty($_GET['items_per_page'])) ? $_GET['items_per_page'] : 0;
        if ($pageSize <= 0) {
            $pageSize = self::$DEFAULT_PAGE_SIZE;
        }
        if ($pageSize >= self::$MAX_DEFAULT_PAGE_SIZE) {
            $pageSize = self::$MAX_DEFAULT_PAGE_SIZE;
        }

        $where = [
            'searchQuery' => trim((!empty($_GET['s'])) ? $_GET['s'] : ''),
        ];
        $orders = $this->ordersManager->getAll($where, [$orderBy, $orderDir], $pageSize, ($currentPage - 1) * $pageSize);
        $rowsCount = $this->ordersManager->countAll($where);
        $this->set_pagination_args(array(
            'total_items' => $rowsCount,
            'per_page' => $pageSize,
        ));

        $statusUpdater = new OrderStatusUpdater();
        foreach ($orders as $polkurierOrder) {
            try {
                if ($statusUpdater->shouldUpdateOrderStatus($polkurierOrder)) {
                    $statusUpdater->updateOrderStatus($polkurierOrder);
                }
                if ($polkurierOrder->getStatusCode() !== 'D') {
                    $polkurierOrder->setExtraValue('delivered_date',  null);
                    $this->ordersManager->save($polkurierOrder);
                }
            } catch (Exception $e) {
            }
        }
        $this->items = $orders;
    }

    public function column_default($item, $column_name): string
    {
        if ($item instanceof Order === false) {
            return '';
        }

        switch ($column_name) {
            case 'mark':
                return '<input type="checkbox" name="label[]" value="' . $item->getOrderNumber() . '"/>';

            case 'order_number':
                return
                    '<a href="post.php?post=' . $item->getShopOrderId() . '&action=edit">' . $item->getOrderNumber() . '</a>'
                    . '<div style="font-size: 80%;">' . $this->getCourierName($item->getCourier()) . '</div>';

            case 'address':
                return $this->formatAddress($item->getSender());

            case 'address_to':
                return $this->formatAddress($item->getRecipient());

            case 'date_send':
                return $item->getExtraValue('pickupdate') ?? '';

            case 'label':
                return $this->formatWaybills($item->getLabel());

            case 'status_date':
                $date = $item->getExtraValue('status_date', '');
                return (!empty($date) && $date !== '1970-01-01 01:00') ? $date : '';

            case 'status':
                return OrderStatusType::getDisplayName($item->getStatusCode()) ?: $item->getStatusCode();

            case 'delivered_date':
                return (string)$item->getExtraValue('delivered_date', '');

            case 'price':
                return $item->getPriceNet() . ' / ' . $item->getPriceGross();

            case 'pobranie':
                return $item->getCod() !== null ? (string)$item->getCod()->amount : '';

            case 'action':
                return $this->column_action($item);

            case 'user':
                if ($item->getAuthorId() !== null) {
                    $user = get_user_by('ID', $item->getAuthorId());
                    if ($user !== null) {
                        return (string)$user->display_name;
                    }
                }
        }

        return '';
    }

    private function getCourierName(string $courierId): string
    {
        try {
            $courier = $this->couriersManager->get($courierId);
            return $courier !== null
                ? $courier->name
                : $courierId;
        } catch (Exception $e) {
            return $courierId;
        }
    }

    private function formatWaybills(array $waybills): string
    {
        $rows = [];
        foreach ($waybills as $waybill) {
            $rows[] .= '<li><span>' . $waybill . '</span></li>';
        }
        return '<ul>' . implode('', $rows) . '</ul>';
    }

    private function formatAddress(?OrderAddress $address): string
    {
        if ($address === null) {
            return '';
        }
        $parts = [];
        $partsExpand = [];

        $parts[] = $address->company;
        $parts[] = $address->person;
        $parts[] = $address->street . " " . $address->housenumber . " " . (!empty($address->flatnumber) ? (' / ' . $address->flatnumber) : '');
        $parts[] = $address->postcode . " " . $address->city . ' (' . ($address->country ?: 'PL') . ')';

        $partsExpand[] = $address->phone;
        $partsExpand[] = $address->email;
        if ($address->point_id !== '') {
            $partsExpand[] = 'Punkt: ' . $address->point_id . ($address->point_name !== '' ? ' (' . $address->point_name . ')' : '');
        }

        return
            implode('<br>', array_filter($parts))
            . (!empty($partsExpand) ? (
                '<div class="pk-expand">'
                . '<div class="pk-expand-content" style="display: none;">'
                . implode('<br>', array_filter($partsExpand))
                . '</div>'
                . '<a href="#" class="pk-expand-button">[Rozwiń]</a>'
                . '</div>'
            ) : '');
    }

    public function column_action(Order $item): string
    {
        return sprintf('%1$s', $this->row_actions([
            'cancel' => sprintf('<a href="?page=%s&action=%s&package=%s&order_id=%s" onclick="return confirm(\'Czy na pewno chcesz anulować zamówienie?\')">Anuluj</a>', $_REQUEST['page'], 'cancel', $item->getOrderNumber(), $item->getShopOrderId()),
            'download' => sprintf('<a href="?page=%s&action=%s&label=%s&order_id=%s" target="_blank">Pobierz etykietę</a>', $_REQUEST['page'], 'label', $item->getOrderNumber(), $item->getShopOrderId()),
            'details' => sprintf('<a href="?page=%s&action=%s&label=%s&order_id=%s" target="_blank">Szczegóły</a>', $_REQUEST['page'], 'details', $item->getOrderNumber(), $item->getShopOrderId()),
        ]));
    }

}
